<?php
/**
 * Bizmart — Company Settings Page.
 *
 * @package Bizmart
 * @license GPL-2.0-or-later
 */

if (!defined('ABSPATH')) exit;

/**
 * Ensure UTF-8 for this page
 */
if (!headers_sent()) {
    header('Content-Type: text/html; charset=UTF-8');
}

// Handle form submission
if (isset($_POST['biz_save_settings'])) {
    $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
    $nonce_ok = ($nonce && wp_verify_nonce($nonce, 'bizmart_company_settings_nonce'));
    if (!$nonce_ok) {
        echo '<div class="notice notice-error is-dismissible"><p>Security check failed.</p></div>';
    } elseif (!current_user_can('manage_woocommerce')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'bizmart'));
    } else {
        // Save all settings (Bizmart-only)
        bizmart_update_option('biz_company_name', sanitize_text_field(wp_unslash($_POST['biz_company_name'] ?? '')));
        bizmart_update_option('biz_company_legal_status', sanitize_text_field(wp_unslash($_POST['biz_company_legal_status'] ?? '')));
        bizmart_update_option('biz_company_slogan', sanitize_text_field(wp_unslash($_POST['biz_company_slogan'] ?? '')));
        bizmart_update_option('biz_company_logo', esc_url_raw(wp_unslash($_POST['biz_company_logo'] ?? '')));

        bizmart_update_option('biz_company_address', sanitize_textarea_field(wp_unslash($_POST['biz_company_address'] ?? '')));
        // Normalize and store phone numbers in E.164-like sanitized form (keep + and digits)
        $raw_phone = wp_unslash($_POST['biz_company_phone'] ?? '');
        $san_phone = preg_replace('/[^+0-9]/', '', (string)$raw_phone);
        bizmart_update_option('biz_company_phone', sanitize_text_field($san_phone));

        $submitted_country = sanitize_text_field(wp_unslash($_POST['biz_company_country'] ?? ''));
        bizmart_update_option('biz_company_country', $submitted_country);
        
        // Sync to WooCommerce
        if (!empty($submitted_country)) {
            update_option('woocommerce_default_country', $submitted_country);
        }

        // City
        $submitted_city = sanitize_text_field(wp_unslash($_POST['biz_company_city'] ?? ''));
        bizmart_update_option('biz_company_city', $submitted_city);
        if (!empty($submitted_city)) {
            update_option('woocommerce_store_city', $submitted_city);
        }

        bizmart_update_option('biz_company_email', sanitize_email(wp_unslash($_POST['biz_company_email'] ?? '')));
        bizmart_update_option('biz_company_website', esc_url_raw(wp_unslash($_POST['biz_company_website'] ?? '')));

        // Social links
        $submitted_instagram = trim(sanitize_text_field(wp_unslash($_POST['biz_company_instagram'] ?? '')));
        $submitted_facebook = trim(sanitize_text_field(wp_unslash($_POST['biz_company_facebook'] ?? '')));
        $submitted_tiktok = trim(sanitize_text_field(wp_unslash($_POST['biz_company_tiktok'] ?? '')));

        // Ensure Instagram/TikTok usernames start with '@' if provided
        if ($submitted_instagram !== '' && strpos($submitted_instagram, '@') !== 0) {
            $submitted_instagram = '@' . $submitted_instagram;
        }
        if ($submitted_tiktok !== '' && strpos($submitted_tiktok, '@') !== 0) {
            $submitted_tiktok = '@' . $submitted_tiktok;
        }

        bizmart_update_option('biz_company_instagram', $submitted_instagram);
        bizmart_update_option('biz_company_facebook', $submitted_facebook);
        bizmart_update_option('biz_company_tiktok', $submitted_tiktok);

        $raw_whatsapp = wp_unslash($_POST['biz_company_whatsapp'] ?? '');
        $san_whatsapp = preg_replace('/[^+0-9]/', '', (string)$raw_whatsapp);
        bizmart_update_option('biz_company_whatsapp', sanitize_text_field($san_whatsapp));

        bizmart_update_option('biz_company_rc', sanitize_text_field(wp_unslash($_POST['biz_company_rc'] ?? '')));
        bizmart_update_option('biz_company_nif', sanitize_text_field(wp_unslash($_POST['biz_company_nif'] ?? '')));
        bizmart_update_option('biz_company_nis', sanitize_text_field(wp_unslash($_POST['biz_company_nis'] ?? '')));
        bizmart_update_option('biz_company_iban', sanitize_text_field(wp_unslash($_POST['biz_company_iban'] ?? '')));

        $display_logo = isset($_POST['biz_display_logo']) ? '1' : '0';
        $display_address = isset($_POST['biz_display_address']) ? '1' : '0';
        $display_legal_numbers = isset($_POST['biz_display_legal_numbers']) ? '1' : '0';
        // New display toggles
        $display_city = isset($_POST['biz_display_city']) ? '1' : '0';
        $display_phone = isset($_POST['biz_display_phone']) ? '1' : '0';
        $display_whatsapp = isset($_POST['biz_display_whatsapp']) ? '1' : '0';
        $display_email = isset($_POST['biz_display_email']) ? '1' : '0';
        $display_social = isset($_POST['biz_display_social']) ? '1' : '0';
        $display_website = isset($_POST['biz_display_website']) ? '1' : '0';

        bizmart_update_option('biz_display_logo', $display_logo);
        bizmart_update_option('biz_display_address', $display_address);
        bizmart_update_option('biz_display_legal_numbers', $display_legal_numbers);
        // Persist new toggles
        bizmart_update_option('biz_display_city', $display_city);
        bizmart_update_option('biz_display_phone', $display_phone);
        bizmart_update_option('biz_display_whatsapp', $display_whatsapp);
        bizmart_update_option('biz_display_email', $display_email);
        bizmart_update_option('biz_display_social', $display_social);
        bizmart_update_option('biz_display_website', $display_website);

        $footer_text = sanitize_textarea_field(wp_unslash($_POST['biz_invoice_footer_text'] ?? ''));
        bizmart_update_option('biz_invoice_footer_text', $footer_text);

        bizmart_update_option('biz_default_currency', sanitize_text_field(wp_unslash($_POST['biz_default_currency'] ?? '')));
        bizmart_update_option('biz_invoice_prefix_format', sanitize_text_field(wp_unslash($_POST['biz_invoice_prefix_format'] ?? '')));

        $admin_language = sanitize_text_field(wp_unslash($_POST['biz_admin_language'] ?? 'en'));
        if (!in_array($admin_language, ['en', 'fr', 'ar'], true)) {
            $admin_language = 'en';
        }
        bizmart_update_option('biz_admin_language', $admin_language);

        // v3.0: Save analytics settings
        bizmart_update_option('biz_margin_alert_threshold', max(0, min(100, absint(wp_unslash($_POST['biz_margin_alert_threshold'] ?? 10)))));
        bizmart_update_option('biz_price_change_alert_pct', max(0, min(100, absint(wp_unslash($_POST['biz_price_change_alert_pct'] ?? 5)))));
        bizmart_update_option('biz_payment_expense_default', isset($_POST['biz_payment_expense_default']) ? '1' : '0');

        // Save expense subcategories
        $sub_json = isset($_POST['biz_expense_subcategories']) ? wp_unslash($_POST['biz_expense_subcategories']) : null;
        if ($sub_json !== null) {
            $subcategories_json = (string) $sub_json;
            $subcategories = json_decode($subcategories_json, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                bizmart_update_option('biz_expense_subcategories', $subcategories);
            }
        }

        $success_message = function_exists('bizmart_text')
            ? bizmart_text('Settings saved successfully!', 'Parametres enregistres avec succes!', 'تم حفظ الإعدادات بنجاح!')
            : 'Settings saved successfully!';
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($success_message) . '</p></div>';
    }
}

// Get current settings
$company_name = bizmart_get_option('biz_company_name', '');
$legal_status = bizmart_get_option('biz_company_legal_status', '');
$slogan = bizmart_get_option('biz_company_slogan', '');
$logo = bizmart_get_option('biz_company_logo', 'https://placeholder.co/150x80/06b6d4/ffffff?text=Logo');

$address = bizmart_get_option('biz_company_address', '');
$company_country = bizmart_get_option('biz_company_country', '');
$company_city = bizmart_get_option('biz_company_city', '');

// Always sync from WooCommerce if available to ensure "changing woo country changes biz too"
if (function_exists('get_option')) {
    $wc_default = get_option('woocommerce_default_country', '');
    $wc_city = get_option('woocommerce_store_city', '');
    if (!empty($wc_default)) {
        if (strpos($wc_default, ':') !== false) {
            list($wc_country, $_) = explode(':', $wc_default, 2);
        } else {
            $wc_country = $wc_default;
        }
        
        // If they differ, update Bizmart to match WooCommerce
        if (!empty($wc_country) && $wc_country !== $company_country) {
            $company_country = $wc_country;
            bizmart_update_option('biz_company_country', $company_country);
        }
    }
    if (!empty($wc_city) && $wc_city !== $company_city) {
        $company_city = $wc_city;
        bizmart_update_option('biz_company_city', $company_city);
    }
}

// Countries list from WooCommerce when available
if (function_exists('WC')) {
    $biz_countries = WC()->countries->countries;
} else {
    $biz_countries = [ 'US' => 'United States', 'FR' => 'France', 'DZ' => 'Algeria', 'GB' => 'United Kingdom' ];
}
$phone = bizmart_get_option('biz_company_phone', '');
$email = bizmart_get_option('biz_company_email', '');
$website = bizmart_get_option('biz_company_website', '');
$instagram = bizmart_get_option('biz_company_instagram', '');
$facebook = bizmart_get_option('biz_company_facebook', '');
$tiktok = bizmart_get_option('biz_company_tiktok', '');
$whatsapp = bizmart_get_option('biz_company_whatsapp', '');

$rc = bizmart_get_option('biz_company_rc', '');
$nif = bizmart_get_option('biz_company_nif', '');
$nis = bizmart_get_option('biz_company_nis', '');
$iban = bizmart_get_option('biz_company_iban', '');

$display_logo = bizmart_get_option('biz_display_logo', '1');
$display_address = bizmart_get_option('biz_display_address', '1');
$display_legal = bizmart_get_option('biz_display_legal_numbers', '1');
$display_city = bizmart_get_option('biz_display_city', '1');
$display_phone = bizmart_get_option('biz_display_phone', '1');
$display_whatsapp = bizmart_get_option('biz_display_whatsapp', '1');
$display_email = bizmart_get_option('biz_display_email', '1');
$display_social = bizmart_get_option('biz_display_social', '1');
$display_website = bizmart_get_option('biz_display_website', '1');
$footer_text = bizmart_get_option('biz_invoice_footer_text', 'Invoice received from (supplier name) by our system Bizmart');

$currency = bizmart_get_option('biz_default_currency', 'DZD');
$invoice_prefix = bizmart_get_option('biz_invoice_prefix_format', 'INV-{YYYY}-{$SEQUENCE}');

$admin_language = bizmart_get_option('biz_admin_language', 'en');
if (!in_array($admin_language, ['en', 'fr', 'ar'], true)) {
    $admin_language = 'en';
}
$html_lang = ($admin_language === 'fr') ? 'fr' : (($admin_language === 'ar') ? 'ar' : 'en');

// Expense subcategories with defaults (Use Unicode escapes for stability across encodings)
$subcategories = bizmart_get_option('biz_expense_subcategories', [
    "\u{1F3E2} Loyer" => ['Bureau', 'Entrepôt', 'Commerce', 'Atelier', 'Parking'],
    "\u{1F465} Salaires" => ['Salaires bruts', 'Charges sociales', 'Primes', 'Heures supplémentaires', 'Formation'],
    "\u{1F4A1} Charges" => ['Électricité', 'Eau', 'Gaz', 'Internet', 'Téléphone', 'Assurances'],
    "\u{1F4E2} Marketing" => ['Publicité en ligne', 'Réseaux sociaux', 'Print', 'Événements', 'SEO/SEM'],
    "\u{1F69A} Livraison" => ['Transport', 'Emballage', 'Frais de port', 'Carburant', 'Entretien véhicules'],
    "\u{1F6E0} Outils & Services" => ['Logiciels', 'Abonnements', 'Maintenance', 'Consulting', 'Comptabilité'],
    "\u{1F4E6} Autre" => ['Fournitures', 'Frais bancaires', 'Taxes', 'Imprévus', 'Divers']
]);

// 1. Define clean metadata for categories (icons and localized labels)
$biz_category_metadata = [
    'Loyer' => [
        'icon' => 'fa-building',
        'en' => 'Rent',
        'fr' => 'Loyer',
        'emoji' => "\u{1F3E2}"
    ],
    'Salaires' => [
        'icon' => 'fa-users',
        'en' => 'Salaries',
        'fr' => 'Salaires',
        'emoji' => "\u{1F465}"
    ],
    'Charges' => [
        'icon' => 'fa-lightbulb',
        'en' => 'Utilities',
        'fr' => 'Charges',
        'emoji' => "\u{1F4A1}"
    ],
    'Marketing' => [
        'icon' => 'fa-bullhorn',
        'en' => 'Marketing',
        'fr' => 'Marketing',
        'emoji' => "\u{1F4E2}"
    ],
    'Livraison' => [
        'icon' => 'fa-truck',
        'en' => 'Delivery',
        'fr' => 'Livraison',
        'emoji' => "\u{1F69A}"
    ],
    'Outils & Services' => [
        'icon' => 'fa-tools',
        'en' => 'Tools & Services',
        'fr' => 'Outils & Services',
        'emoji' => "\u{1F6E0}"
    ],
    'Autre' => [
        'icon' => 'fa-box',
        'en' => 'Other',
        'fr' => 'Autre',
        'emoji' => "\u{1F4E6}"
    ],
];

// 2. Pre-process $subcategories to recover from mojibake (corrupted encoding)
$recovered_subcategories = [];
foreach ($subcategories as $raw_category => $subs) {
    $matched_key = null;
    $norm = strtolower((string)$raw_category);
    
    // Fuzzy matching to clean keys even with typos or corruption
    if (strpos($norm, 'loyer') !== false) $matched_key = 'Loyer';
    elseif (strpos($norm, 'salaires') !== false || strpos($norm, 'slaires') !== false) $matched_key = 'Salaires';
    elseif (strpos($norm, 'charges') !== false) $matched_key = 'Charges';
    elseif (strpos($norm, 'marketing') !== false) $matched_key = 'Marketing';
    elseif (strpos($norm, 'livraison') !== false) $matched_key = 'Livraison';
    elseif (strpos($norm, 'outils') !== false) $matched_key = 'Outils & Services';
    elseif (strpos($norm, 'autre') !== false) $matched_key = 'Autre';
    
    // Use the matched clean key if found, otherwise keep original (stripping suspected junk)
    $final_key = $matched_key ? $matched_key : trim(preg_replace('/[^a-zA-Z0-9 &]/', '', $raw_category));
    if (!$final_key) $final_key = $raw_category; // Fallback
    
    $recovered_subcategories[$final_key] = $subs;
}
$subcategories = $recovered_subcategories;

$subcategory_display_labels = [
    'Loyer' => [
        'Bureau' => "\u{1F3E2} " . (function_exists('bizmart_text') ? bizmart_text('Office', 'Bureau', 'المكتب') : 'Office'),
        'Entrepot' => "\u{1F3ED} " . (function_exists('bizmart_text') ? bizmart_text('Warehouse', 'Entrepot', 'المستودع') : 'Warehouse'),
        'Commerce' => "\u{1F6CD} " . (function_exists('bizmart_text') ? bizmart_text('Shop', 'Commerce', 'المتجر') : 'Shop'),
        'Atelier' => "\u{1F3D7} " . (function_exists('bizmart_text') ? bizmart_text('Workshop', 'Atelier', 'الورشة') : 'Workshop'),
        'Parking' => "\u{1F17F} " . (function_exists('bizmart_text') ? bizmart_text('Parking', 'Parking', 'موقف سيارات') : 'Parking'),
    ],
    'Salaires' => [
        'Salaires bruts' => "\u{1F4B5} " . (function_exists('bizmart_text') ? bizmart_text('Gross salaries', 'Salaires bruts', 'الرواتب الإجمالية') : 'Gross salaries'),
        'Charges sociales' => "\u{1F3DB} " . (function_exists('bizmart_text') ? bizmart_text('Social charges', 'Charges sociales', 'الاشتراكات الاجتماعية') : 'Social charges'),
        'Primes' => "\u{1F381} " . (function_exists('bizmart_text') ? bizmart_text('Bonuses', 'Primes', 'العلاوات') : 'Bonuses'),
        'Heures supplementaires' => "\u{1F237} " . (function_exists('bizmart_text') ? bizmart_text('Overtime', 'Heures supplementaires', 'العمل الإضافي') : 'Overtime'),
        'Formation' => "\u{1F393} " . (function_exists('bizmart_text') ? bizmart_text('Training', 'Formation', 'التدريب') : 'Training'),
    ],
    'Charges' => [
        'Electricite' => "\u{26A1} " . (function_exists('bizmart_text') ? bizmart_text('Electricity', 'Electricite', 'كهرباء') : 'Electricity'),
        'Eau' => "\u{1F4A7} " . (function_exists('bizmart_text') ? bizmart_text('Water', 'Eau', 'مياه') : 'Water'),
        'Gaz' => "\u{1F525} " . (function_exists('bizmart_text') ? bizmart_text('Gas', 'Gaz', 'غاز') : 'Gas'),
        'Internet' => "\u{1F310} " . (function_exists('bizmart_text') ? bizmart_text('Internet', 'Internet', 'إنترنت') : 'Internet'),
        'Telephone' => "\u{1F4DE} " . (function_exists('bizmart_text') ? bizmart_text('Phone', 'Telephone', 'هاتف') : 'Telephone'),
        'Assurances' => "\u{1F6E1} " . (function_exists('bizmart_text') ? bizmart_text('Insurance', 'Assurances', 'تأمين') : 'Insurance'),
    ],
    'Marketing' => [
        'Publicite en ligne' => "\u{1F5B1} " . (function_exists('bizmart_text') ? bizmart_text('Online advertising', 'Publicite en ligne', 'إعلان عبر الإنترنت') : 'Online advertising'),
        'Reseaux sociaux' => "\u{1F4F1} " . (function_exists('bizmart_text') ? bizmart_text('Social media', 'Reseaux sociaux', 'وسائل التواصل الاجتماعي') : 'Social media'),
        'Print' => "\u{1F4E0} " . (function_exists('bizmart_text') ? bizmart_text('Print', 'Print', 'طباعة') : 'Print'),
        'Evenements' => "\u{1F389} " . (function_exists('bizmart_text') ? bizmart_text('Events', 'Evenements', 'الأحداث') : 'Events'),
        'SEO/SEM' => "\u{1F50D} " . (function_exists('bizmart_text') ? bizmart_text('SEO/SEM', 'SEO/SEM', 'تحسين محركات البحث/التسويق') : 'SEO/SEM'),
    ],
    'Livraison' => [
        'Transport' => "\u{1F69A} " . (function_exists('bizmart_text') ? bizmart_text('Transport', 'Transport', 'النقل') : 'Transport'),
        'Emballage' => "\u{1F4E6} " . (function_exists('bizmart_text') ? bizmart_text('Emballage', 'Emballage', 'التغليف') : 'Emballage'),
        'Frais de port' => "\u{2709} " . (function_exists('bizmart_text') ? bizmart_text('Shipping fees', 'Frais de port', 'رسوم الشحن') : 'Shipping fees'),
        'Carburant' => "\u{26FD} " . (function_exists('bizmart_text') ? bizmart_text('Fuel', 'Carburant', 'الوقود') : 'Fuel'),
        'Entretien vehicules' => "\u{1F527} " . (function_exists('bizmart_text') ? bizmart_text('Vehicle maintenance', 'Entretien vehicules', 'صيانة المركبات') : 'Vehicle maintenance'),
    ],
    'Outils & Services' => [
        'Logiciels' => "\u{1F4BB} " . (function_exists('bizmart_text') ? bizmart_text('Software', 'Logiciels', 'البرمجيات') : 'Software'),
        'Abonnements' => "\u{1F4B3} " . (function_exists('bizmart_text') ? bizmart_text('Subscriptions', 'Abonnements', 'الاشتراكات') : 'Subscriptions'),
        'Maintenance' => "\u{1F527} " . (function_exists('bizmart_text') ? bizmart_text('Maintenance', 'Maintenance', 'الصيانة') : 'Maintenance'),
        'Consulting' => "\u{1F91D} " . (function_exists('bizmart_text') ? bizmart_text('Consulting', 'Consulting', 'استشارات') : 'Consulting'),
        'Comptabilite' => "\u{1F4CA} " . (function_exists('bizmart_text') ? bizmart_text('Accounting', 'Comptabilite', 'المحاسبة') : 'Accounting'),
    ],
    'Autre' => [
        'Fournitures' => "\u{1F4CE} " . (function_exists('bizmart_text') ? bizmart_text('Supplies', 'Fournitures', 'المستلزمات') : 'Supplies'),
        'Frais bancaires' => "\u{1F3DB} " . (function_exists('bizmart_text') ? bizmart_text('Bank fees', 'Frais bancaires', 'رسوم بنكية') : 'Bank fees'),
        'Taxes' => "\u{1F4DC} " . (function_exists('bizmart_text') ? bizmart_text('Taxes', 'Taxes', 'الضرائب') : 'Taxes'),
        'Imprevus' => "\u{26A0} " . (function_exists('bizmart_text') ? bizmart_text('Unexpected', 'Imprevus', 'غير متوقع') : 'Unexpected'),
        'Divers' => "\u{1F300} " . (function_exists('bizmart_text') ? bizmart_text('Miscellaneous', 'Divers', 'متفرقات') : 'Miscellaneous'),
    ],
];

// Import presets (purchase price / cost meta) - shown in Settings importer panel
$biz_import_presets = [
    [
        'slug' => 'cost-of-goods-for-woocommerce/cost-of-goods-for-woocommerce.php',
        'name' => 'Cost of Goods',
        'author' => 'WPFactory',
        'meta_key' => '_alg_wc_cog_cost',
    ],
    [
        'slug' => 'ni-woocommerce-cost-of-goods/ni-woocommerce-cost-of-goods.php',
        'name' => 'Cost of Goods',
        'author' => 'Ni',
        'meta_key' => '_ni_cost_goods',
    ],
    // Generic/commonly-used key (may come from themes or custom code)
    [
        'slug' => null,
        'name' => 'WC cost meta',
        'author' => 'WooCommerce / theme',
        'meta_key' => '_wc_cost',
    ],
];

$biz_active_plugins = (array) get_option('active_plugins', []);
$biz_detected_presets = [];
$biz_other_presets = [];
foreach ($biz_import_presets as $p) {
    $slug = $p['slug'] ?? null;
    if ($slug && in_array($slug, $biz_active_plugins, true)) {
        $biz_detected_presets[] = $p;
    } else {
        $biz_other_presets[] = $p;
    }
}
?>

    <?php
    // Use system font stack instead of Google Fonts to avoid GDPR / external resource concerns.
    // If you wish to use Plus Jakarta Sans, self-host the font files instead.
    wp_enqueue_style('bizmart-fontawesome', BIZMART_CORE_URL . 'assets/vendor/fontawesome.min.css', [], '6.0.0');
    ?>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            /* Refined Professional Palette */
            --primary: #4f46e5;       /* Indigo 600 */
            --primary-hover: #4338ca; /* Indigo 700 */
            --primary-soft: #e0e7ff;  /* Indigo 100 */
            --secondary: #0ea5e9;     /* Sky 500 */
            
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            
            --bg-body: #f8fafc;       /* Slate 50 */
            --bg-surface: #ffffff;
            
            --text-main: #0f172a;     /* Slate 900 */
            --text-secondary: #64748b; /* Slate 500 */
            --text-tertiary: #94a3b8;
            
            --border-color: #e2e8f0;
            
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --radius-md: 0.5rem;
            --radius-lg: 0.75rem;
            
            /* Legacy mappings */
            --color-dark: var(--text-main);
            --color-text-light: var(--text-secondary);
            --color-bg-soft: var(--primary-soft);
            --color-border-light: var(--border-color);
            --color-accent: var(--secondary);
        }

        body {
            font-family: 'Plus Jakarta Sans', -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Segoe UI Emoji', 'Apple Color Emoji', 'Noto Color Emoji', Roboto, sans-serif;
            background-color: var(--bg-body);
            color: var(--text-main);
            min-height: 100vh;
            padding: 0; /* keep WP admin layout intact; plugin content gets its own padding */
            line-height: 1.5;
        }

        /* Subtle ambient background */
        body::before {
            content: '';
            position: fixed;
            top: 0; left: 0; right: 0;
            height: 400px;
            background: radial-gradient(circle at 50% 0%, #e0e7ff 0%, transparent 70%);
            z-index: -1;
            pointer-events: none;
            opacity: 0.6;
        }

        .biz-modern-container {
            /* Use more horizontal space on wide screens but stay padded */
            max-width: 1400px;
            margin: 24px auto;
            padding: 1rem 24px; /* keep breathing room */
        }

        .biz-modern-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2.5rem;
            padding-bottom: 1.5rem;
            border-bottom: 1px solid rgba(0,0,0,0.05);
        }

        .biz-modern-header h1 {
            font-size: 2rem;
            font-weight: 800;
            color: var(--text-main);
            margin: 0;
            letter-spacing: -0.025em;
        }
        
        .biz-modern-header p {
            color: var(--text-secondary);
            font-size: 0.95rem;
            margin-top: 0.25rem;
        }

        .biz-save-float {
            position: fixed;
            bottom: 2rem;
            right: 2rem;
            z-index: 999999; /* ensure it's above other admin UI */
            pointer-events: auto;
        }

        /* When a WP media modal is open, hide the floating save to avoid overlapping controls */
        .biz-save-float.biz-media-open {
            display: none !important;
            pointer-events: none !important;
            opacity: 0.6;
        }

        .biz-btn-save {
            background: var(--text-main);
            color: white;
            border: none;
            padding: 0.875rem 2rem;
            border-radius: 9999px;
            font-weight: 700;
            font-size: 1rem;
            cursor: pointer;
            pointer-events: auto;
            /* explicit shadow for reliability */
            box-shadow: 0 10px 20px rgba(0,0,0,0.12), 0 4px 6px rgba(0,0,0,0.08);
            transition: all 0.2s;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .biz-btn-save:hover {
            transform: translateY(-2px);
            background: #000;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }

        .biz-tabs-container {
            display: grid;
            /* slightly wider sidebar to allow content to use remaining width */
            grid-template-columns: 320px 1fr;
            gap: 24px;
            align-items: start;
        }

        .biz-nav-sidebar {
            position: sticky;
            top: 2rem;
            background: var(--bg-surface);
            border-radius: var(--radius-lg);
            padding: 1rem;
            box-shadow: var(--shadow-sm);
            border: 1px solid var(--border-color);
            width: 320px;
        }

        .biz-nav-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            margin-bottom: 0.25rem;
            border-radius: var(--radius-md);
            color: var(--text-secondary);
            font-weight: 600;
            font-size: 0.95rem;
            cursor: pointer;
            transition: all 0.2s ease;
            border: 1px solid transparent;
        }

        .biz-nav-item i {
            font-size: 1.1em;
            width: 1.5rem;
            text-align: center;
            color: var(--text-tertiary);
            transition: color 0.2s;
        }

        .biz-nav-item:hover {
            background-color: var(--bg-body);
            color: var(--text-main);
        }
        
        .biz-nav-item:hover i {
            color: var(--primary);
        }

        .biz-nav-item.active {
            background-color: var(--primary-soft);
            color: var(--primary);
        }

        .biz-nav-item.active i {
            transform: none;
            color: var(--primary);
        }

        .biz-content-area {
            min-height: 500px;
        }

        .biz-section {
            display: none;
            animation: fadeIn 0.3s ease-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .biz-section.active {
            display: block;
        }

        /* Section titles: slightly reduced for better balance with labels */
        .biz-section-title {
            font-size: 1.25rem; /* reduce from 1.5rem */
            font-weight: 700;
            margin-bottom: 0.5rem;
            color: var(--text-main);
            line-height: 1.2;
        }

        /* Subtitles should be clearly secondary and smaller than labels */
        .biz-section-subtitle {
            color: var(--text-secondary);
            font-size: 0.85rem; /* smaller than labels */
            margin-bottom: 1rem; /* tighten spacing */
            line-height: 1.3;
        }

        .biz-card-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.25rem;
            margin-bottom: 1.5rem;
            align-items: stretch;
        }

        .biz-card {
            background: var(--bg-surface);
            border-radius: var(--radius-lg);
            padding: 1.5rem;
            border: 1px solid var(--border-color);
            box-shadow: var(--shadow-sm);
            transition: all 0.2s ease;
            /* Make fields flow into columns when space permits */
            display: grid;
            grid-template-columns: 1fr;
            gap: 1rem;
        }

        .biz-card:hover {
            border-color: #cbd5e1;
            box-shadow: var(--shadow-md);
            transform: translateY(-1px);
        }

        .biz-field-group {
            margin-bottom: 1.25rem;
        }

        .biz-field-label {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-weight: 600;
            color: var(--text-main);
            margin-bottom: 0.5rem;
            font-size: 0.95rem; /* slightly larger than section subtitle */
        }

        .biz-field-label i {
            color: var(--text-tertiary);
            font-size: 0.9em;
        }

        .biz-input, .biz-textarea, .biz-select {
            width: 100%;
            padding: 0.625rem 0.875rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 0.95rem;
            color: var(--text-main);
            transition: all 0.2s ease;
            font-family: inherit;
            background: #fff;
            box-sizing: border-box;
        }

        /* Intl-tel-input responsive fixes */
        .iti { width: 100%; display: block; }
        .iti input { width: 100% !important; }
        .iti__country-list { z-index: 1000; }

        /* Responsive contact fields grid */
        .biz-contact-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 1.25rem;
            width: 100%;
        }

        /* Display toggles grid */
        .biz-display-grid {
            display: grid;
            grid-template-columns: repeat(3, minmax(220px, 1fr));
            gap: 1rem 1.5rem;
            align-items: start;
            margin-bottom: 0.75rem;
        }

        .biz-input:focus, .biz-textarea:focus, .biz-select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px var(--primary-soft);
        }

        .biz-textarea {
            min-height: 100px;
            resize: vertical;
        }

        /* Responsive tweaks -------------------------------------------------- */
        @media (max-width: 900px) {
            .biz-modern-container {
                margin: 12px 12px;
                padding: 0.75rem;
                max-width: none;
            }

            .biz-tabs-container {
                grid-template-columns: 1fr; /* stack sidebar above content */
                gap: 12px;
            }

            .biz-nav-sidebar {
                position: relative;
                top: auto;
                display: flex;
                gap: 8px;
                overflow-x: auto;
                padding: 0.5rem;
            }

            .biz-nav-item { flex: 0 0 auto; }

            .biz-card-grid { grid-template-columns: 1fr; }

            .biz-card { grid-template-columns: 1fr; }
        }

        /* Responsive for display toggles */
        @media (max-width: 900px) {
            .biz-display-grid { grid-template-columns: 1fr; }
        }

        /* Logo preview styles */
        .biz-logo-preview {
            width: 150px;
            height: 80px;
            object-fit: contain;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            background: #fff;
            padding: 6px;
        }

        @media (min-width: 900px) {
            .biz-card { grid-template-columns: 1fr; }
            .biz-card-grid { grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); }
        }

        @media (min-width: 1400px) {
            .biz-modern-container { max-width: 1600px; }
            .biz-card-grid { grid-template-columns: repeat(3, 1fr); }
            .biz-card { grid-template-columns: repeat(2, 1fr); }
        }

        /* Compact inline toggle presentation */
        .biz-toggle-group {
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 12px;
            padding: 6px 4px;
            margin-bottom: 6px;
        }

        /* Larger label and icon, smaller neutral toggle (non-iOS) */
        .biz-toggle-label { font-weight:600; font-size:1rem; color:var(--text-main); display:flex; flex-direction:column; align-items:flex-start; gap:6px; flex:1; }
        .biz-toggle-label .biz-toggle-main { display:flex; align-items:center; gap:10px; }
        .biz-toggle-label i { font-size:16px; color:var(--text-tertiary); }
        .biz-toggle-note { font-size:0.85rem; color:var(--text-secondary); margin-top:2px; }

        .biz-toggle { width:36px; height:20px; border-radius:12px; position:relative; background:#f1f5f9; border:1px solid var(--border-color); transition:background .12s ease, border-color .12s; flex:0 0 auto; }
        .biz-toggle .biz-toggle-knob { position:absolute; top:2px; left:2px; width:16px; height:16px; background:#fff; border-radius:50%; box-shadow:0 1px 2px rgba(0,0,0,.06); transition:left .12s ease, box-shadow .12s; }
        .biz-toggle.on { background:var(--primary); border-color:var(--primary); }
        .biz-toggle.on .biz-toggle-knob { left:18px; box-shadow:0 2px 4px rgba(0,0,0,.12); }

        @media (max-width:700px) { .biz-display-grid { grid-template-columns: repeat(2, 1fr); } }

        .biz-logo-preview {
            width: 120px;
            height: 120px;
            border-radius: 16px;
            object-fit: cover;
            border: 3px solid var(--gray-200);
            margin-bottom: 12px;
        }

        .biz-btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            transition: all 0.2s;
            border: 1px solid transparent;
            text-decoration: none;
            line-height: 1.25;
        }

        .biz-btn-primary {
            background: var(--primary);
            color: white;
            box-shadow: 0 1px 2px 0 rgba(0,0,0,0.05);
        }

        .biz-btn-primary:hover {
            background: var(--primary-hover);
            transform: translateY(-1px);
            box-shadow: var(--shadow-sm);
        }

        .biz-btn-secondary {
            background: white;
            border: 1px solid var(--border-color);
            color: var(--text-secondary);
        }

        .biz-btn-secondary:hover {
            border-color: #cbd5e1;
            color: var(--text-main);
            background: var(--bg-body);
        }

        .description {
            font-size: 0.85rem;
            color: var(--text-secondary);
            margin-top: 0.4rem;
        }

        /* Match the exact style of .biz-field-label for visual consistency */
        .biz-global-toggle-note {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.95rem; /* match field label size */
            color: var(--text-main);
            margin-top: 6px;
            font-weight: 600;
            background: transparent;
            padding: 0;
            border-radius: 6px;
            border: none;
        }
        .biz-global-toggle-note i { color: var(--text-tertiary); font-size: 0.95rem; }

        /* Subsection title used for compact subsections inside cards - match field labels */
        .biz-subsection-title {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.95rem; /* match field labels */
            font-weight: 600;
            color: var(--text-main);
            margin-bottom: 0.5rem;
        }
        .biz-subsection-title i { color: var(--text-tertiary); font-size: 0.95rem; }

        @media (max-width: 1024px) {
            .biz-tabs-container {
                grid-template-columns: 1fr;
            }
            .biz-nav-sidebar {
                position: static;
                display: flex;
                overflow-x: auto;
                gap: 8px;
            }
            .biz-nav-item {
                white-space: nowrap;
                margin-bottom: 0;
            }
        }

        @media (max-width: 768px) {
            .biz-modern-header {
                flex-direction: column;
                gap: 20px;
                text-align: center;
            }
            .biz-card-grid {
                grid-template-columns: 1fr;
            }
        }




        .prefix-segment-tag {
            display: inline-flex;
            align-items: center;
            background-color: var(--bg-body);
            color: var(--text-main);
            padding: 0.25rem 0.75rem;
            margin: 0.25rem;
            border-radius: 9999px;
            font-size: 0.85rem;
            font-weight: 600;
            cursor: pointer;
            border: 1px solid var(--border-color);
            transition: all 0.2s;
        }
        .prefix-segment-tag:hover {
            background-color: var(--primary-soft);
            color: var(--primary);
            border-color: var(--primary-soft);
        }

        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.6);
            display: none;
            justify-content: center;
            align-items: center;
            z-index: 100000;
        }
        .modal-content {
            background: #fff;
            padding: 0;
            border-radius: 12px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
            width: 90%;
            max-width: 800px;
            max-height: 90vh;
            overflow-y: auto;
            position: relative;
        }
        .invoice-preview-body {
            padding: 30px;
            border: 1px solid #f0f0f1;
            box-shadow: 0 0 10px rgba(0,0,0,0.05);
        }
        .close-button {
            position: absolute;
            top: 10px;
            right: 15px;
            font-size: 24px;
            cursor: pointer;
            color: #ccc;
            line-height: 1;
            z-index: 10;
        }
        .close-button:hover {
            color: var(--color-accent);
        }
        
        @media print {
            body * {
                visibility: hidden;
            }
            .invoice-preview-body, .invoice-preview-body * {
                visibility: visible;
            }
            .invoice-preview-body {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
            }
            .close-button {
                display: none;
            }
        }
    </style>
</head>
<body>

<!-- MODAL D'APERCU DE FACTURE -->
<div id="invoice-preview-modal" class="modal-overlay">
    <div class="modal-content">
        <span id="close-modal-btn" class="close-button">&times;</span>
        <div class="invoice-preview-body">
            <h3 class="text-2xl font-bold text-gray-800 mb-6"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Preview Test Invoice', 'Apercu de la Facture Test', 'معاينة فاتورة تجريبية') : 'Preview Test Invoice'); ?></h3>
            
            <!-- En-tete de Facture -->
            <div class="flex justify-between items-start mb-10 border-b pb-4">
                <div class="space-y-1">
                    <img id="invoice-logo-preview" src="<?php echo esc_url($logo); ?>" alt="Logo" class="w-32 h-auto object-contain">
                    <p id="invoice-company-name" class="font-bold text-lg text-gray-800"><?php echo esc_html($company_name); ?></p>
                    <p id="invoice-company-address" class="text-sm text-gray-600"><?php echo nl2br(esc_html($address)); ?></p>
                    <p class="text-xs text-gray-500 mt-2"><span id="invoice-reg-numbers">RC: <?php echo esc_html($rc); ?> | NIF: <?php echo esc_html($nif); ?> | NIS: <?php echo esc_html($nis); ?></span></p>
                </div>
                <div class="text-right">
                    <h4 class="text-cyan-600 font-extrabold text-3xl mb-2"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('INVOICE', 'FACTURE', 'فاتورة') : 'INVOICE'); ?></h4>
                    <p class="text-sm font-semibold"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Date:', 'Date :', 'التاريخ:') : 'Date:'); ?> <?php echo esc_html(wp_date('d/m/Y')); ?></p>
                    <p class="text-sm font-semibold"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Invoice #:', 'NÂ° de Facture :', 'رقم الفاتورة:') : 'Invoice #:'); ?></p>
                    <p id="invoice-number-display" class="text-xl font-mono text-blue-700 bg-blue-50 p-1 rounded-md mt-1">INV-2025-00042</p>
                </div>
            </div>

            <!-- Infos Client -->
            <div class="mb-10 p-4 bg-gray-50 rounded-lg">
                <p class="font-semibold text-gray-700"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('BILLED FROM:', 'FACTURE PAR :', 'مفوترة من:') : 'BILLED FROM:'); ?></p>
                <p class="text-sm"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Example Supplier', 'Exemple Fournisseur', 'مورد مثال') : 'Example Supplier'); ?><br><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Supplier Address', 'Adresse du Fournisseur', 'عنوان المورد') : 'Supplier Address'); ?><br><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('City, Country', 'Ville, Pays', 'المدينة، الدولة') : 'City, Country'); ?></p>
            </div>

            <!-- Postes -->
            <table class="w-full text-left border-collapse mb-10">
                <thead>
                    <tr class="text-sm border-b-2 border-cyan-300 text-cyan-700">
                        <th class="py-2">Description</th>
                        <th class="py-2 w-20 text-right"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Qty', 'Qte', 'الكمية') : 'Qty'); ?></th>
                        <th class="py-2 w-32 text-right"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Unit Price', 'Prix Unitaire', 'سعر الوحدة') : 'Unit Price'); ?></th>
                        <th class="py-2 w-32 text-right"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Amount', 'Montant', 'المبلغ') : 'Amount'); ?> (<?php echo esc_html($currency); ?>)</th>
                    </tr>
                </thead>
                <tbody>
                    <tr class="border-b border-gray-200">
                        <td class="py-2"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Example product 1', 'Produit Exemple 1', 'مثال منتج 1') : 'Example product 1'); ?></td>
                        <td class="py-2 text-right">1</td>
                        <td class="py-2 text-right">50 000,00</td>
                        <td class="py-2 text-right">50 000,00</td>
                    </tr>
                    <tr class="border-b border-gray-200">
                        <td class="py-2"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Example product 2', 'Produit Exemple 2', 'مثال منتج 2') : 'Example product 2'); ?></td>
                        <td class="py-2 text-right">2</td>
                        <td class="py-2 text-right">25 000,00</td>
                        <td class="py-2 text-right">50 000,00</td>
                    </tr>
                </tbody>
                <tfoot>
                    <tr class="text-sm font-semibold">
                        <td colspan="3" class="py-2 text-right"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Subtotal', 'Sous-total', 'المجموع الفرعي') : 'Subtotal'); ?></td>
                        <td class="py-2 text-right">100 000,00</td>
                    </tr>
                    <tr class="text-sm font-semibold">
                        <td colspan="3" class="py-2 text-right"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Tax (9%)', 'TVA (9%)', 'الضريبة (9%)') : 'Tax (9%)'); ?></td>
                        <td class="py-2 text-right">9 000,00</td>
                    </tr>
                    <tr class="font-bold text-lg text-white bg-blue-600">
                        <td colspan="3" class="py-3 px-3 text-right rounded-bl-lg"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('TOTAL DUE', 'TOTAL DU', 'الإجمالي المستحق') : 'TOTAL DUE'); ?> (<?php echo esc_html($currency); ?>)</td>
                        <td class="py-3 px-3 text-right rounded-br-lg">109 000,00</td>
                    </tr>
                </tfoot>
            </table>

            <!-- Pied de Page -->
            <div class="text-center pt-4 border-t border-gray-300">
                <p id="invoice-footer-display" class="text-sm text-gray-700 italic"><?php echo esc_html($footer_text ?: 'Invoice received from (supplier name) by our system Bizmart'); ?></p>
                <p class="text-xs text-gray-500 mt-2">IBAN : <?php echo esc_html($iban); ?></p>
            </div>
            
            <div class="mt-6 text-center no-print">
                <button onclick="window.print()" class="button-primary"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Print', 'Imprimer', 'طباعة') : 'Print'); ?></button>
            </div>
        </div>
    </div>
</div>

<div class="biz-modern-container">
    <!-- Modern Header -->
    <div class="biz-modern-header">
        <div>
            <h1><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Company Settings', "Parametres de l'Entreprise") : 'Company Settings'); ?></h1>
            <p style="margin:8px 0 0 0; color: var(--gray-600); font-size: 14px;">
                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Manage your business information and preferences', 'Gerez les informations de votre entreprise', 'إدارة معلومات وتفضيلات شركتك') : 'Manage your business information and preferences'); ?>
            </p>
        </div>
        <div style="display: flex; gap: 12px; align-items: center;">
            <i class="fas fa-building" style="font-size: 40px; color: var(--primary);"></i>
        </div>
    </div>

    <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=bizmart-settings')); ?>" id="biz-modern-form">
        <?php 
        wp_enqueue_media();
        wp_enqueue_script('jquery');
        wp_nonce_field('bizmart_company_settings_nonce'); 
        ?>

        <!-- Tab Navigation + Content -->
        <div class="biz-tabs-container">
            <!-- Sidebar Navigation -->
            <div class="biz-nav-sidebar">
                <div class="biz-nav-item active" data-section="general">
                    <i class="fas fa-building"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('General', 'General', 'عام') : 'General'); ?></span>
                </div>
                <div class="biz-nav-item" data-section="contact">
                    <i class="fas fa-address-card"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Contact', 'Contact', 'الاتصال') : 'Contact'); ?></span>
                </div>
                <div class="biz-nav-item" data-section="legal">
                    <i class="fas fa-file-contract"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Legal', 'Juridique', 'قانوني') : 'Legal'); ?></span>
                </div>
                <div class="biz-nav-item" data-section="display">
                    <i class="fas fa-eye"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Display', 'Affichage', 'العرض') : 'Display'); ?></span>
                </div>
                <div class="biz-nav-item" data-section="defaults">
                    <i class="fas fa-sliders-h"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Defaults', 'Par defaut', 'الإعدادات الافتراضية') : 'Defaults'); ?></span>
                </div>
                <div class="biz-nav-item" data-section="analytics">
                    <i class="fas fa-chart-line"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Analytics', 'Analytique', 'التحليلات') : 'Analytics'); ?></span>
                </div>
                <div class="biz-nav-item" data-section="expenses">
                    <i class="fas fa-receipt"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Expenses', 'Depenses', 'المصاريف') : 'Expenses'); ?></span>
                </div>
                <div class="biz-nav-item" data-section="import">
                    <i class="fas fa-file-import"></i>
                    <span><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Import/Export', 'Import/Export', 'استيراد/تصدير') : 'Import/Export'); ?></span>
                </div>
            </div>

            <!-- Content Area -->
            <div class="biz-content-area">
                <!-- General Section -->
                <div class="biz-section active" data-section="general">
                    <h2 class="biz-section-title">
                        <i class="fas fa-building" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Company Identity', "Identite de l'Entreprise", 'هوية الشركة') : 'Company Identity'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Basic information about your company', 'Informations de base sur votre entreprise', 'معلومات أساسية عن شركتك') : 'Basic information about your company'); ?></p>

                    <div class="biz-card-grid">
                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-building"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Company Name', "Nom de l'Entreprise", 'اسم الشركة') : 'Company Name'); ?>
                                </div>
                                <input type="text" name="biz_company_name" id="company-name-input" value="<?php echo esc_attr($company_name); ?>" placeholder="<?php echo esc_attr(function_exists('bizmart_text') ? bizmart_text('Your company name', 'Nom de votre entreprise', 'اسم شركتك') : 'Your company name'); ?>" class="biz-input" required>
                            </div>
                        </div>

                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-balance-scale"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Legal Status', 'Statut Juridique', 'الوضع القانوني') : 'Legal Status'); ?>
                                </div>
                                <input type="text" name="biz_company_legal_status" value="<?php echo esc_attr($legal_status); ?>" class="biz-input" placeholder="SARL, SPA, etc.">
                            </div>
                        </div>

                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-quote-right"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Slogan', 'Slogan', 'الشعار') : 'Slogan'); ?>
                                </div>
                                <input type="text" name="biz_company_slogan" value="<?php echo esc_attr($slogan); ?>" class="biz-input" placeholder="<?php echo esc_attr(function_exists('bizmart_text') ? bizmart_text('Your business slogan', 'Votre slogan', 'شعار عملك') : 'Your business slogan'); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Logo Upload Card -->
                    <div class="biz-card">
                        <div class="biz-field-label">
                            <i class="fas fa-image"></i>
                            <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Company Logo', 'Logo de l\'Entreprise', 'شعار الشركة') : 'Company Logo'); ?>
                        </div>
                        <div style="display: flex; gap: 24px; align-items: center;">
                            <?php
                                // Always render the image element so JS can update it
                                $default_placeholder = 'https://placeholder.co/150x80/06b6d4/ffffff?text=Logo';
                                $show_image = !empty($logo) && strpos($logo, 'placeholder.co') === false && $logo !== $default_placeholder;
                            ?>
                            <img id="logo-preview-img" 
                                 src="<?php echo esc_url($logo); ?>" 
                                 alt="<?php echo esc_attr($company_name ?: 'Logo'); ?>" 
                                 class="biz-logo-preview" 
                                 style="<?php echo esc_attr($show_image ? '' : 'display:none;'); ?>">
                            <div style="flex: 1;">
                                <button type="button" id="upload-logo-btn" class="biz-btn biz-btn-primary" style="display: flex; align-items: center; gap: 8px;">
                                    <i class="fas fa-upload"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Upload Logo', 'Telecharger Logo', 'تحميل الشعار') : 'Upload Logo'); ?>
                                </button>
                                <p style="margin-top: 12px; font-size: 13px; color: var(--gray-500);">
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Recommended: 200x100px, transparent background', 'Recommande: 200x100px, fond transparent', 'الموصى به: 200x100px، خلفية شفافة') : 'Recommended: 200x100px, transparent background'); ?>
                                </p>
                            </div>
                        </div>
                        <input type="hidden" name="biz_company_logo" id="logo-url-input" value="<?php echo esc_url($logo); ?>">
                    </div>
                </div>

                <!-- Contact Section -->
                <div class="biz-section" data-section="contact">
                    <h2 class="biz-section-title">
                        <i class="fas fa-address-card" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Contact Information', 'Coordonnees', 'معلومات الاتصال') : 'Contact Information'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('How customers can reach you', 'Comment les clients peuvent vous contacter', 'كيفية تواصل العملاء معك') : 'How customers can reach you'); ?></p>

                    <div class="biz-card-grid">
                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-flag"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Country', 'Pays', 'البلد') : 'Country'); ?>
                                </div>
                                <select name="biz_company_country" class="biz-select">
                                    <option value="">-- <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Select country', 'Selectionner un pays', 'اختر البلد') : 'Select country'); ?> --</option>
                                    <?php foreach($biz_countries as $code => $label): ?>
                                        <option value="<?php echo esc_attr($code); ?>" <?php selected($company_country, $code); ?>><?php echo esc_html($label); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-city"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('City', 'Ville', 'المدينة') : 'City'); ?>
                                </div>
                                <div id="biz-city-container">
                                    <input type="text" name="biz_company_city" value="<?php echo esc_attr($company_city); ?>" class="biz-input" placeholder="<?php echo esc_attr(function_exists('bizmart_text') ? bizmart_text('City', 'Ville', 'المدينة') : 'City'); ?>">
                                </div>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-map-marker-alt"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Address', 'Adresse', 'العنوان') : 'Address'); ?>
                                </div>
                                <textarea name="biz_company_address" id="address-input" class="biz-textarea" placeholder="<?php echo esc_attr(function_exists('bizmart_text') ? bizmart_text('Full business address', 'Adresse complete', 'العنوان الكامل للأعمال') : 'Full business address'); ?>"><?php echo esc_textarea($address); ?></textarea>
                            </div>
                        </div>

                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-phone"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Phone', 'Telephone', 'الهاتف') : 'Phone'); ?>
                                </div>
                                <input type="tel" name="biz_company_phone" value="<?php echo esc_attr($phone); ?>" class="biz-input" placeholder="+213 XXX XX XX XX" data-initial-country="<?php echo esc_attr($company_country); ?>">
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fab fa-whatsapp"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('WhatsApp', 'WhatsApp', 'واتساب') : 'WhatsApp'); ?>
                                </div>
                                <input type="tel" name="biz_company_whatsapp" value="<?php echo esc_attr($whatsapp); ?>" class="biz-input" placeholder="+213 XXX XX XX XX">
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-envelope"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Email', 'Email', 'البريد الإلكتروني') : 'Email'); ?>
                                </div>
                                <input type="email" name="biz_company_email" value="<?php echo esc_attr($email); ?>" class="biz-input" placeholder="contact@company.com">
                            </div>
                        </div>

                        <div class="biz-card">
                                <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fab fa-facebook"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Facebook', 'Facebook', 'فيسبوك') : 'Facebook'); ?>
                                </div>
                                <input type="text" name="biz_company_facebook" value="<?php echo esc_attr($facebook); ?>" class="biz-input" placeholder="facebook.com/yourpage">
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fab fa-instagram"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Instagram', 'Instagram', 'انستغرام') : 'Instagram'); ?>
                                </div>
                                <input type="text" name="biz_company_instagram" value="<?php echo esc_attr($instagram); ?>" class="biz-input" placeholder="@yourhandle">
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fab fa-tiktok"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('TikTok', 'TikTok', 'تيك توك') : 'TikTok'); ?>
                                </div>
                                <input type="text" name="biz_company_tiktok" value="<?php echo esc_attr($tiktok); ?>" class="biz-input" placeholder="@yourhandle">
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-globe"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Website', 'Site Web', 'الموقع الإلكتروني') : 'Website'); ?>
                                </div>
                                <input type="url" name="biz_company_website" value="<?php echo esc_attr($website); ?>" class="biz-input" placeholder="https://www.example.com">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Legal Section -->
                <div class="biz-section" data-section="legal">
                    <h2 class="biz-section-title">
                        <i class="fas fa-file-contract" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Legal Information', 'Informations Legales', 'المعلومات القانونية') : 'Legal Information'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Registration numbers and banking details', 'Numeros d\'enregistrement et coordonnees bancaires') : 'Registration numbers and banking details'); ?></p>

                    <div class="biz-card-grid">
                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-file-alt"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('RC (Trade Register)', 'RC (Registre de Commerce)', 'السجل التجاري (RC)') : 'RC (Trade Register)'); ?>
                                </div>
                                <input type="text" name="biz_company_rc" id="rc-input" value="<?php echo esc_attr($rc); ?>" class="biz-input">
                            </div>
                        </div>

                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-hashtag"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('NIF (Tax ID)', 'NIF (ID Fiscal)', 'الرقم الضريبي (NIF)') : 'NIF (Tax ID)'); ?>
                                </div>
                                <input type="text" name="biz_company_nif" id="nif-input" value="<?php echo esc_attr($nif); ?>" class="biz-input">
                            </div>
                        </div>

                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-id-card"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('NIS (Social Security Number)', 'NIS (Numero Securite Sociale)', 'الرقم الاجتماعي (NIS)') : 'NIS (Social Security Number)'); ?>
                                </div>
                                <input type="text" name="biz_company_nis" id="nis-input" value="<?php echo esc_attr($nis); ?>" class="biz-input">
                            </div>
                        </div>

                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-university"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('IBAN (Bank Account)', 'IBAN (Compte Bancaire)', 'آيبان (حساب بنكي)') : 'IBAN (Bank Account)'); ?>
                                </div>
                                <input type="text" name="biz_company_iban" value="<?php echo esc_attr($iban); ?>" class="biz-input" placeholder="DZ00 0000 0000 0000 0000">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Display Section -->
                <div class="biz-section" data-section="display">
                    <h2 class="biz-section-title">
                        <i class="fas fa-eye" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Display Preferences', "Preferences d'Affichage") : 'Display Preferences'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Control what appears on invoices and documents', 'Controlez ce qui apparait sur les factures', 'تحكم فيما يظهر على الفواتير والوثائق') : 'Control what appears on invoices and documents'); ?></p>

                    <div class="biz-card">
                        <div class="biz-field-group">
                            <div class="biz-subsection-title">
                                <i class="fas fa-language"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Admin Language', 'Langue Admin', 'لغة الإدارة') : 'Admin Language'); ?>
                            </div>
                            <select name="biz_admin_language" class="biz-select">
    <option value="en" <?php selected($admin_language, 'en'); ?>>English</option>
    <option value="fr" <?php selected($admin_language, 'fr'); ?>>Français</option>
    <option value="ar" <?php selected($admin_language, 'ar'); ?>>العربية</option>
</select>
                            <p style="margin-top: 8px; font-size: 13px; color: var(--gray-500);">
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Language used throughout the Bizmart admin interface', 'Langue utilisee dans l\'interface Bizmart') : 'Language used throughout the Bizmart admin interface'); ?>
                            </p>
                            <div class="biz-global-toggle-note">
                                <i class="fas fa-language" aria-hidden="true"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show / Hide information from invoices', 'Afficher / Masquer les informations sur les factures', 'إظهار/إخفاء المعلومات من الفواتير') : 'Show / Hide information from invoices'); ?>
                            </div>
                        </div>

                        <div class="biz-display-grid">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-image"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show Logo on Invoices', 'Afficher le Logo', 'عرض الشعار على الفواتير') : 'Show Logo on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_logo" <?php checked($display_logo, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-map-marker-alt"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show Address on Invoices', 'Afficher l\'Adresse') : 'Show Address on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_address" <?php checked($display_address, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-file-alt"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show Legal Numbers on Invoices', 'Afficher Numeros Legaux', 'عرض الأرقام القانونية على الفواتير') : 'Show Legal Numbers on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_legal_numbers" <?php checked($display_legal, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-city"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show City on Invoices', 'Afficher la Ville', 'عرض المدينة على الفواتير') : 'Show City on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_city" <?php checked($display_city, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-phone"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show Phone on Invoices', 'Afficher Telephone', 'عرض الهاتف على الفواتير') : 'Show Phone on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_phone" <?php checked($display_phone, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fab fa-whatsapp"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show WhatsApp on Invoices', 'Afficher WhatsApp', 'عرض واتساب على الفواتير') : 'Show WhatsApp on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_whatsapp" <?php checked($display_whatsapp, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-envelope"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show Email on Invoices', 'Afficher Email', 'عرض البريد الإلكتروني على الفواتير') : 'Show Email on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_email" <?php checked($display_email, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-share-alt"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show Social Links on Invoices', 'Afficher les reseaux sociaux', 'عرض روابط التواصل الاجتماعي على الفواتير') : 'Show Social Links on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_social" <?php checked($display_social, '1'); ?>>
                            </div>

                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-globe"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Show Website on Invoices', 'Afficher Site Web', 'عرض الموقع الإلكتروني على الفواتير') : 'Show Website on Invoices'); ?>
                                </div>
                                <input type="checkbox" name="biz_display_website" <?php checked($display_website, '1'); ?>>
                            </div>
                        </div>

                        <div class="biz-field-group">
                            <div class="biz-subsection-title">
                                <i class="fas fa-align-left"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Invoice Footer Text', 'Texte Pied de Page', 'نص تذييل الفاتورة') : 'Invoice Footer Text'); ?>
                            </div>
                            <textarea name="biz_invoice_footer_text" id="footer-text-input" class="biz-textarea"><?php echo esc_textarea($footer_text); ?></textarea>
                            <p style="margin-top: 8px; font-size: 13px; color: var(--gray-500);">
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('This text will appear at the bottom of every invoice', 'Ce texte apparaitra en bas de chaque facture', 'سيظهر هذا النص أسفل كل فاتورة') : 'This text will appear at the bottom of every invoice'); ?>
                            </p>
                        </div>
                    </div>
                </div>

                <!-- Defaults Section -->
                <div class="biz-section" data-section="defaults">
                    <h2 class="biz-section-title">
                        <i class="fas fa-sliders-h" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Default Settings', 'Parametres par Defaut', 'الإعدادات الافتراضية') : 'Default Settings'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Configure default values for invoices and documents', 'Configurez les valeurs par defaut', 'تكوين القيم الافتراضية للفواتير والوثائق') : 'Configure default values for invoices and documents'); ?></p>

                    <div class="biz-card">
                        <div class="biz-field-group">
                            <div class="biz-field-label">
                                <i class="fas fa-coins"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Default Currency', 'Devise par Defaut', 'العملة الافتراضية') : 'Default Currency'); ?>
                            </div>
                            <input type="text" name="biz_default_currency" value="<?php echo esc_attr($currency); ?>" class="biz-input" style="max-width: 150px;" placeholder="DZD">
                            <p style="margin-top: 8px; font-size: 13px; color: var(--gray-500);">
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Currency code used for all financial operations', 'Code devise utilise pour toutes les operations', 'رمز العملة المستخدم في جميع العمليات المالية') : 'Currency code used for all financial operations'); ?>
                            </p>
                        </div>

                        <div class="biz-field-group">
                            <div class="biz-field-label">
                                <i class="fas fa-hashtag"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Invoice Number Format', 'Format Numero de Facture', 'تنسيق رقم الفاتورة') : 'Invoice Number Format'); ?>
                            </div>
                            <div style="display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 12px;">
                                <button type="button" class="biz-btn biz-btn-secondary" style="font-size: 12px;" data-segment="INV">INV-</button>
                                <button type="button" class="biz-btn biz-btn-secondary" style="font-size: 12px;" data-segment="SUPPLIER">SUPPLIER</button>
                                <button type="button" class="biz-btn biz-btn-secondary" style="font-size: 12px;" data-segment="YEAR">YYYY</button>
                                <button type="button" class="biz-btn biz-btn-secondary" style="font-size: 12px;" data-segment="DATE">YYMMDD</button>
                                <button type="button" class="biz-btn biz-btn-secondary" style="font-size: 12px;" data-segment="SEQUENCE">SEQUENCE</button>
                            </div>
                            <div id="prefix-segments-container" style="border: 2px dashed var(--gray-300); border-radius: 10px; padding: 12px; min-height: 50px; margin-bottom: 12px;"></div>
                            <input type="text" name="biz_invoice_prefix_format" id="invoice-prefix-input" value="<?php echo esc_attr($invoice_prefix); ?>" class="biz-input" style="font-family: monospace;">
                            <p id="prefix-output" style="margin-top: 12px; font-size: 14px; color: var(--primary); font-weight: 600; font-family: monospace;">
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Example: INV-2025-00042', 'Exemple: INV-2025-00042', 'مثال: INV-2025-00042') : 'Example: INV-2025-00042'); ?>
                            </p>
                        </div>
                    </div>
                </div>
                <!-- v3.0: Analytics Section -->
                <div class="biz-section" data-section="analytics">
                    <h2 class="biz-section-title">
                        <i class="fas fa-chart-line" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Analytics & Alerts', 'Analytique & Alertes', 'التحليلات والتنبيهات') : 'Analytics & Alerts'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Configure margin alerts and payment defaults', 'Configurez les alertes de marge et paiements par défaut', 'إعداد تنبيهات الهامش والدفعات الافتراضية') : 'Configure alerts and defaults'); ?></p>

                    <?php
                    $cur_margin_th = bizmart_get_option('biz_margin_alert_threshold', '10');
                    $cur_price_pct = bizmart_get_option('biz_price_change_alert_pct', '5');
                    $cur_payment_default = bizmart_get_option('biz_payment_expense_default', '1');
                    ?>

                    <div style="display:grid; grid-template-columns: 1fr 1fr; gap:16px; margin-bottom:20px;">
                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-exclamation-triangle" style="color:#f59e0b;"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Margin Alert Threshold (%)', 'Seuil Alerte Marge (%)', 'حد تنبيه الهامش (%)') : 'Margin Alert Threshold (%)'); ?>
                                </div>
                                <p style="font-size:12px; color:var(--gray-500); margin:0 0 8px;">
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Products below this margin will appear in alerts', 'Les produits sous cette marge apparaitront dans les alertes', 'المنتجات تحت هذا الهامش ستظهر في التنبيهات') : 'Products below this margin appear in alerts'); ?>
                                </p>
                                <div style="display:flex; align-items:center; gap:8px;">
                                    <input type="number" name="biz_margin_alert_threshold" value="<?php echo esc_attr($cur_margin_th); ?>" min="0" max="100" step="1" class="biz-input" style="max-width:100px;">
                                    <span style="font-size:18px; font-weight:600; color:var(--gray-500);">%</span>
                                </div>
                            </div>
                        </div>
                        <div class="biz-card">
                            <div class="biz-field-group">
                                <div class="biz-field-label">
                                    <i class="fas fa-chart-line" style="color:#ef4444;"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Price Change Alert (%)', 'Alerte Variation Prix (%)', 'تنبيه تغير السعر (%)') : 'Price Change Alert (%)'); ?>
                                </div>
                                <p style="font-size:12px; color:var(--gray-500); margin:0 0 8px;">
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Alert when price changes more than this %', 'Alerte quand le prix change de plus de ce %', 'تنبيه عندما يتغير السعر بأكثر من هذه النسبة') : 'Alert on price change exceeding this %'); ?>
                                </p>
                                <div style="display:flex; align-items:center; gap:8px;">
                                    <input type="number" name="biz_price_change_alert_pct" value="<?php echo esc_attr($cur_price_pct); ?>" min="0" max="100" step="1" class="biz-input" style="max-width:100px;">
                                    <span style="font-size:18px; font-weight:600; color:var(--gray-500);">%</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="biz-card" style="margin-top:16px;">
                        <div class="biz-field-group">
                            <div class="biz-field-label">
                                <i class="fas fa-money-check-alt" style="color:#059669;"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Default: Add payment to expenses', 'Par défaut: Ajouter le paiement aux dépenses', 'افتراضي: إضافة الدفعة إلى المصروفات') : 'Default: Add payment to expenses'); ?>
                            </div>
                            <p style="font-size:12px; color:var(--gray-500); margin:0 0 10px;">
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('When adding a payment, the "Add to expenses" checkbox will be checked by default', 'Lors de l\'ajout d\'un paiement, la case "Ajouter aux dépenses" sera cochée par défaut', 'عند إضافة دفعة، سيتم تحديد خانة "إضافة إلى المصروفات" افتراضيًا') : 'Payment expense checkbox default'); ?>
                            </p>
                            <label style="display:flex; align-items:center; gap:10px; cursor:pointer;">
                                <input type="checkbox" name="biz_payment_expense_default" value="1" <?php checked($cur_payment_default, '1'); ?> style="width:18px; height:18px; accent-color:var(--primary);">
                                <span style="font-size:14px; font-weight:500; color:var(--gray-700);">
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Checked by default', 'Coché par défaut', 'محدد افتراضيًا') : 'Checked by default'); ?>
                                </span>
                            </label>
                        </div>
                    </div>
                </div>

                <!-- Expenses Section -->
                <div class="biz-section" data-section="expenses">
                    <h2 class="biz-section-title">
                        <i class="fas fa-receipt" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Expense Categories', 'Categories de Depenses', 'فئات المصروفات') : 'Expense Categories'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Manage and customize expense subcategories', 'Gerez les sous-categories de depenses', 'إدارة وتخصيص الفئات الفرعية للمصروفات') : 'Manage and customize expense subcategories'); ?></p>

                    <input type="hidden" name="biz_expense_subcategories" id="biz-subcategories-data" value="<?php echo esc_attr(json_encode($subcategories)); ?>">
                    
                    <div id="subcategories-manager" style="display: grid; gap: 20px;">
                        <?php foreach ($subcategories as $category => $subs): ?>
                        <div class="biz-card" data-category="<?php echo esc_attr($category); ?>">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 16px;">
                                <div class="biz-field-label" style="margin: 0; font-size: 16px;">
                                    <?php 
                                    $meta = $biz_category_metadata[$category] ?? null;
                                    if ($meta) {
                                        echo '<i class="fas ' . esc_attr($meta['icon']) . '" style="color: var(--primary); margin-right: 8px;"></i>';
                                        $label = function_exists('bizmart_text') ? bizmart_text($meta['en'], $meta['fr']) : $meta['fr'];
                                        $emoji = $meta['emoji'] ?? '';
                                        echo esc_html(($emoji ? $emoji . ' ' : '') . $label);
                                    } else {
                                        // Fallback for custom categories
                                        echo esc_html($category);
                                    }
                                    ?>
                                </div>
                                <button type="button" class="biz-btn biz-btn-primary add-subcat-btn" data-category="<?php echo esc_attr($category); ?>" style="font-size: 12px; padding: 8px 16px;">
                                    <i class="fas fa-plus"></i>
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Add', 'Ajouter', 'إضافة') : 'Add'); ?>
                                </button>
                            </div>
                            <div class="subcategories-list" style="display: flex; flex-wrap: wrap; gap: 8px;">
                                <?php foreach ($subs as $sub): ?>
                                <div class="subcat-item" style="display: flex; align-items: center; gap: 8px; background: var(--gray-50); border: 1px solid var(--gray-200); border-radius: 20px; padding: 6px 14px;">
                                    <?php $sub_display = $subcategory_display_labels[$category][$sub] ?? $sub; ?>
                                    <span class="subcat-text" style="font-size: 13px;" data-subcat-value="<?php echo esc_attr($sub); ?>"><?php echo esc_html($sub_display); ?></span>
                                    <button type="button" class="delete-subcat-btn" style="border: none; background: none; color: var(--danger); cursor: pointer; font-size: 18px; line-height: 1;" title="<?php echo esc_attr(function_exists('bizmart_text') ? bizmart_text('Delete', 'Supprimer', 'حذف') : 'Delete'); ?>">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- Import/Export Section -->
                <div class="biz-section" data-section="import">
                    <h2 class="biz-section-title">
                        <i class="fas fa-file-import" style="color: var(--primary); margin-right: 12px;"></i>
                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Import Product Prices', 'Importer Prix Produits', 'استيراد أسعار المنتجات') : 'Import Product Prices'); ?>
                    </h2>
                    <p class="biz-section-subtitle"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Import purchase prices from other plugins', 'Importez les prix d\'achat') : 'Import purchase prices from other plugins'); ?></p>

                    <div class="biz-card">
                        <?php if (!empty($biz_detected_presets)): ?>
                        <div style="background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); border: 2px solid #10b981; border-radius: 12px; padding: 16px; margin-bottom: 20px;">
                            <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 8px;">
                                <i class="fas fa-check-circle" style="color: #10b981; font-size: 24px;"></i>
                                <strong style="color: #065f46; font-size: 15px;">Detected purchase price sources on this site</strong>
                            </div>
                            <div style="display: flex; gap: 8px; flex-wrap: wrap;">
                                <?php foreach ($biz_detected_presets as $p): ?>
                                    <span style="display: inline-flex; align-items: center; gap: 8px; background: white; border: 1px solid #10b981; color: #065f46; padding: 8px 14px; border-radius: 999px; font-size: 13px; font-weight: 600;">
                                        <i class="fas fa-plug"></i>
                                        <?php echo esc_html($p['name'] . ' - ' . $p['author']); ?>
                                    </span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <div class="biz-field-group">
                            <div class="biz-field-label">
                                <i class="fas fa-database"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Select Source', 'Selectionner Source', 'اختر المصدر') : 'Select Source'); ?>
                            </div>
                            <select id="import-meta-preset" class="biz-select">
                                <option value="">Import purchase prices from known plugins...</option>
                                <option value="_custom">Enter custom meta key...</option>
                                <?php if (!empty($biz_detected_presets)): ?>
                                    <optgroup label="Detected on this site">
                                        <?php foreach ($biz_detected_presets as $p): ?>
                                            <option value="<?php echo esc_attr($p['meta_key']); ?>"><?php echo esc_html($p['name'] . ' - ' . $p['author'] . ' (' . $p['meta_key'] . ')'); ?></option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php endif; ?>
                                <optgroup label="All available presets">
                                    <?php foreach ($biz_import_presets as $p): ?>
                                        <option value="<?php echo esc_attr($p['meta_key']); ?>"><?php echo esc_html($p['name'] . ' - ' . $p['author'] . ' (' . $p['meta_key'] . ')'); ?></option>
                                    <?php endforeach; ?>
                                </optgroup>
                            </select>
                            <div id="biz-preset-detected-indicator" style="display:none;margin-top:12px;padding:10px;background:var(--gray-50);border-radius:8px;color:var(--success);font-weight:600;">
                                <i class="fas fa-check-circle"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Purchase prices detected on this site', 'Prix detectes sur ce site', 'أسعار الشراء المكتشفة على هذا الموقع') : 'Purchase prices detected on this site'); ?>
                            </div>
                        </div>

                        <div class="biz-field-group">
                            <div class="biz-field-label">
                                <i class="fas fa-key"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Custom Meta Key', 'Cle Meta Personnalisee', 'مفتاح ميتا مخصص') : 'Custom Meta Key'); ?>
                            </div>
                            <input type="text" id="import-meta-key" class="biz-input" placeholder="e.g. _alg_wc_cog_cost or _wc_cost,_custom_cost" list="bizmart-import-meta-datalist">
                            <datalist id="bizmart-import-meta-datalist">
                                <?php foreach ($biz_import_presets as $p): ?>
                                    <option value="<?php echo esc_attr($p['meta_key']); ?>"></option>
                                <?php endforeach; ?>
                            </datalist>
                            <p style="margin-top: 8px; font-size: 13px; color: var(--gray-500);">
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Enter comma-separated keys to try multiple sources in order', 'Entrez plusieurs cles separees par des virgules', 'أدخل مفاتيح مفصولة بفواصل لتجربة مصادر متعددة بالترتيب') : 'Enter comma-separated keys to try multiple sources in order'); ?>
                            </p>
                        </div>

                        <div class="biz-field-group">
                            <div class="biz-field-label">
                                <i class="fas fa-cog"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Conflict Policy', 'Politique de Conflit', 'سياسة التعارض') : 'Conflict Policy'); ?>
                            </div>
                            <select id="import-policy" class="biz-select">
                                <option value="prompt">Prompt (pause on conflicts)</option>
                                <option value="replace">Replace existing values</option>
                                <option value="skip">Skip existing values</option>
                            </select>
                        </div>

                        <div style="display: flex; gap: 12px; flex-wrap: wrap;">
                            <button type="button" id="biz-get-import-count" class="biz-btn biz-btn-secondary">
                                <i class="fas fa-calculator"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Get Count', 'Compter', 'احصل على العدد') : 'Get Count'); ?>
                            </button>
                            <button type="button" id="biz-start-import" class="biz-btn biz-btn-primary">
                                <i class="fas fa-play"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Start Import', 'Demarrer Import', 'بدء الاستيراد') : 'Start Import'); ?>
                            </button>
                            <button type="button" id="biz-export-import-audit" class="biz-btn biz-btn-secondary">
                                <i class="fas fa-download"></i>
                                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Download Audit CSV', 'Telecharger Audit', 'تحميل سجل التدقيق CSV') : 'Download Audit CSV'); ?>
                            </button>
                        </div>

                        <div id="biz-import-summary" style="display:none; margin-top:20px; border:2px solid var(--primary); border-radius:12px; padding:16px; background:var(--gray-50);">
                    <div style="display:flex;gap:12px;flex-wrap:wrap;align-items:center;justify-content:space-between;">
                        <div>
                            <div style="font-weight:700; color:#111827;">Import progress</div>
                            <div id="biz-import-summary-text" class="description" style="margin-top:2px;">-</div>
                        </div>
                        <div style="min-width:240px; flex:1; max-width:420px;">
                            <div style="height:10px;background:#e5e7eb;border-radius:999px;overflow:hidden;">
                                <div id="biz-import-progress-bar" style="height:10px;width:0%;background:linear-gradient(90deg,#06b6d4,#1d4ed8);"></div>
                            </div>
                            <div id="biz-import-progress-meta" style="margin-top:6px;font-size:12px;color:#6b7280;display:flex;justify-content:space-between;gap:10px;">
                                <span id="biz-import-progress-left">0%</span>
                                <span id="biz-import-progress-right">0 / 0</span>
                            </div>
                        </div>
                    </div>
                </div>

                <div id="biz-import-resume-area" style="display:none; margin-top:10px;">
                    <div class="description">Job is paused for confirmation due to conflicts. Choose an action:</div>
                    <div class="flex gap-3 mt-2">
                        <button type="button" id="biz-resume-replace-all" class="button-primary">Replace all</button>
                        <button type="button" id="biz-resume-skip-all" class="button-secondary">Skip all</button>
                    </div>
                </div>

                <div id="biz-import-conflicts-area" style="display:none; margin-top:10px;">
                    <div class="description">Resolve conflicts (existing purchase price vs source). Choose Replace/Skip per item, then submit decisions.</div>
                    <div class="mt-2" style="overflow:auto; max-height: 320px; border: 1px solid #e5e7eb; border-radius: 8px;">
                        <table class="w-full text-left border-collapse">
                            <thead>
                                <tr class="text-sm border-b border-gray-200 bg-gray-50">
                                    <th class="py-2 px-3">Product</th>
                                    <th class="py-2 px-3">SKU</th>
                                    <th class="py-2 px-3">Existing</th>
                                    <th class="py-2 px-3">Incoming</th>
                                    <th class="py-2 px-3">Selling</th>
                                    <th class="py-2 px-3">Decision</th>
                                </tr>
                            </thead>
                            <tbody id="biz-import-conflicts-body"></tbody>
                        </table>
                    </div>
                    <div class="flex gap-3 mt-3">
                        <button type="button" id="biz-submit-decisions" class="button-primary">Submit decisions</button>
                    </div>
                </div>

                <div id="biz-import-preview-area" style="display:none; margin-top:10px;">
                    <div class="description">Recent processed items (preview)</div>
                    <div class="mt-2" style="overflow:auto; max-height: 240px; border: 1px solid #e5e7eb; border-radius: 8px;">
                        <table class="w-full text-left border-collapse">
                            <thead>
                                <tr class="text-sm border-b border-gray-200 bg-gray-50">
                                    <th class="py-2 px-3">Product</th>
                                    <th class="py-2 px-3">SKU</th>
                                    <th class="py-2 px-3">Type</th>
                                    <th class="py-2 px-3">Incoming</th>
                                    <th class="py-2 px-3">Existing</th>
                                    <th class="py-2 px-3">Selling</th>
                                    <th class="py-2 px-3">Action</th>
                                </tr>
                            </thead>
                            <tbody id="biz-import-preview-body"></tbody>
                        </table>
                    </div>
                </div>

                <div id="biz-import-available-area" style="display:none; margin-top:10px;">
                    <div style="display:flex;justify-content:space-between;gap:12px;align-items:flex-end;flex-wrap:wrap;">
                        <div>
                            <div style="font-weight:700;color:#111827;">Available products for this meta key</div>
                            <div id="biz-import-available-note" class="description" style="margin-top:2px;">-</div>
                        </div>
                        <div style="min-width:220px;">
                            <label style="font-size:12px;color:#6b7280;">Filter</label>
                            <input type="text" id="biz-import-available-filter" class="biz-input" placeholder="Search by name, SKU, ID" style="max-width:260px;">
                        </div>
                    </div>
                    <div class="mt-2" style="overflow:auto; max-height: 360px; border: 1px solid #e5e7eb; border-radius: 8px; background:#fff;">
                        <table class="w-full text-left border-collapse">
                            <thead>
                                <tr class="text-sm border-b border-gray-200 bg-gray-50">
                                    <th class="py-2 px-3">Product</th>
                                    <th class="py-2 px-3">SKU</th>
                                    <th class="py-2 px-3">Type</th>
                                    <th class="py-2 px-3">Meta raw</th>
                                    <th class="py-2 px-3">Normalized</th>
                                    <th class="py-2 px-3">Existing</th>
                                    <th class="py-2 px-3">Selling</th>
                                </tr>
                            </thead>
                            <tbody id="biz-import-available-body"></tbody>
                        </table>
                    </div>
                </div>

                            <div id="biz-import-status" style="font-size: 14px; font-weight: 600;"></div>
                            <details style="margin-top:12px;">
                                <summary style="cursor:pointer; color: var(--gray-700); font-weight:600; padding: 8px 0;">
                                    <i class="fas fa-code"></i> Technical details (JSON)
                                </summary>
                                <textarea id="biz-import-json" rows="6" class="biz-textarea" readonly style="font-family: monospace; margin-top: 12px;"></textarea>
                            </details>
                        </div>

                        <!-- Cleanup Section -->
                        <div style="margin-top:24px; border-top:2px solid var(--gray-200); padding-top:20px;">
                            <div style="background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); border: 2px solid var(--danger); border-radius: 12px; padding: 16px;">
                                <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 12px;">
                                    <i class="fas fa-exclamation-triangle" style="color: var(--danger); font-size: 20px;"></i>
                                    <strong style="color: #991b1b; font-size: 15px;"><?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Danger Zone - Delete Source Meta', 'Zone Dangereuse', 'منطقة الخطر - حذف ميتا المصدر') : 'Danger Zone'); ?></strong>
                                </div>
                                <p style="color: #7f1d1d; font-size: 13px; margin-bottom: 12px;">
                                    <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Permanently delete source meta values. This is irreversible.', 'Suppression permanente des meta-valeurs. Irreversible.', 'حذف قيم الميتا للمصدر نهائيًا. هذا عمل لا يمكن التراجع عنه.') : 'Permanently delete source meta values. This is irreversible.'); ?>
                                </p>
                                <div style="display: flex; gap: 10px;">
                                    <button type="button" id="biz-start-cleanup" class="biz-btn" style="background: var(--danger); color: white;">
                                        <i class="fas fa-trash-alt"></i>
                                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Start Cleanup', 'Demarrer Nettoyage', 'بدء التنظيف') : 'Start Cleanup'); ?>
                                    </button>
                                    <button type="button" id="biz-export-cleanup-audit" class="biz-btn biz-btn-secondary">
                                        <i class="fas fa-download"></i>
                                        <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Download Cleanup Audit', 'Telecharger Audit', 'تحميل سجل التنظيف') : 'Download Cleanup Audit'); ?>
                                    </button>
                                </div>

                                <div id="biz-cleanup-summary" style="display:none; margin-top:16px; border:2px solid #991b1b; border-radius:10px; padding:12px; background: white;">
                                    <div style="font-weight:700; color:#7f1d1d; margin-bottom: 8px;">Cleanup Progress</div>
                                    <div id="biz-cleanup-status" style="font-size: 13px; color: var(--gray-600); margin-bottom: 10px;"></div>
                                    <div style="height:10px;background:#fee2e2;border-radius:999px;overflow:hidden;">
                                        <div id="biz-cleanup-progress-bar" style="height:10px;width:0%;background:linear-gradient(90deg,#fca5a5,#ef4444);transition: width 0.3s ease;"></div>
                                    </div>
                                    <div style="margin-top:8px;font-size:12px;color:#6b7280;display:flex;justify-content:space-between;">
                                        <span id="biz-cleanup-pct">0%</span>
                                        <span id="biz-cleanup-fraction">0 / 0</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

            </div> <!-- end content area -->
        </div> <!-- end tabs container -->

    </form>
</div> <!-- end modern container -->

<script>
jQuery(document).ready(function($) {
    const adminLang = '<?php echo esc_js($admin_language); ?>';
    const ajaxNonce = '<?php echo esc_js(wp_create_nonce("bizmart-analytics-nonce")); ?>';
    const ajaxUrl = '<?php echo esc_url(admin_url("admin-ajax.php")); ?>';
    const bizDetectedMetaKeys = <?php echo wp_json_encode(array_column($biz_detected_presets, 'meta_key')); ?>;
    const editPostUrlBase = '<?php echo esc_url(admin_url("post.php")); ?>';
    const t = (en, fr, ar) => {
        if (adminLang === 'ar') {
            const dict = (window.BizmartI18n && BizmartI18n.ar) ? BizmartI18n.ar : null;
            return ar || (dict && dict[en]) || en;
        }
        return adminLang === 'fr' ? fr : en;
    };

    // Media Uploader for Logo
    (function($){
        var image_frame = null;
        $('#upload-logo-btn').on('click', function(e){
            e.preventDefault();
            if (typeof wp === 'undefined' || !wp.media) {
                alert('Media uploader is not available. Please refresh the page.');
                return;
            }
            if (image_frame) {
                image_frame.open();
                return;
            }
            image_frame = wp.media({
                title: t('Select or upload a logo', 'Selectionner ou telecharger un logo'),
                multiple: false,
                library: { type: 'image' }
            });

            // When media modal opens, hide/disable the floating save so it doesn't cover modal controls
            image_frame.on('open', function(){
                try { $('.biz-save-float').addClass('biz-media-open').hide(); } catch(e){}
            });

            image_frame.on('close', function(){
                try { $('.biz-save-float').removeClass('biz-media-open').show(); } catch(e){}
            });

            image_frame.on('select', function(){
                var attachment = image_frame.state().get('selection').first().toJSON();
                $('#logo-url-input').val(attachment.url);
                $('#logo-preview-img').attr('src', attachment.url).show();
                // close the modal after selecting to ensure clean state
                try { image_frame.close(); } catch(e){}
            });

            image_frame.open();
        });
    })(jQuery);
    
    // Invoice Prefix Format Builder
    const prefixOutput = document.getElementById('prefix-output');
    const prefixSegmentsContainer = document.getElementById('prefix-segments-container');
    const prefixInput = document.getElementById('invoice-prefix-input');
    const modal = document.getElementById('invoice-preview-modal');

    // Manage segments as an ordered array to append left->right
    let segments = [];
    if (prefixInput && prefixInput.value) {
        segments = prefixInput.value.split('-').filter(s => s.trim());
    }

    function renderSegments() {
        prefixSegmentsContainer.innerHTML = '';
        segments.forEach(seg => {
            const tag = document.createElement('span');
            tag.className = 'prefix-segment-tag';
            let label = seg;
            if (seg === 'INV' || seg === 'INV-') label = 'INV';
            else if (seg === '{SUPPLIER}') label = '{SUPPLIER}';
            else if (seg === '{YYYY}') label = t('{YYYY}', '{AAAA}');
            else if (seg === '{YYMMDD}') label = t('{YYMMDD}', '{AAMMJJ}');
            else if (seg === '{$SEQUENCE}') label = '{$SEQUENCE}';
            tag.textContent = label.replace(/-$/, '');
            tag.title = seg;
            prefixSegmentsContainer.appendChild(tag);
        });
        prefixInput.value = segments.join('-');
        updatePreview();
    }

    function addSegment(segmentId) {
        const map = { 'INV':'INV', 'SUPPLIER':'{SUPPLIER}', 'YEAR':'{YYYY}', 'DATE':'{YYMMDD}', 'SEQUENCE':'{$SEQUENCE}' };
        const token = map[segmentId];
        if (!token) return;

        // Ensure only one sequence and that it's always at the end
        const hadSequence = segments.includes('{$SEQUENCE}');
        if (hadSequence) segments = segments.filter(s => s !== '{$SEQUENCE}');

        // Prevent duplicate non-sequence tokens
        if (token !== '{$SEQUENCE}' && !segments.includes(token)) {
            segments.push(token);
        }

        // If adding sequence, ensure it's last
        if (token === '{$SEQUENCE}') {
            // already removed earlier
            segments.push('{$SEQUENCE}');
        } else if (hadSequence) {
            segments.push('{$SEQUENCE}');
        }

        renderSegments();
    }

    function getPreviewNumber() {
        if (!segments || segments.length === 0) return '';
        const parts = segments.map(s => {
            if (s === 'INV') return 'INV';
            if (s === '{SUPPLIER}') return 'SUP01';
            if (s === '{YYYY}') return new Date().getFullYear();
            if (s === '{YYMMDD}') return '251209';
            if (s === '{$SEQUENCE}') return '00042';
            return s;
        });
        return parts.join('-');
    }

    function updatePreview() {
        prefixOutput.textContent = `${t('Example:', 'Exemple :')} ${getPreviewNumber()}`;
    }

    function showModal() {
        document.getElementById('invoice-number-display').textContent = getPreviewNumber();
        const footerVal = $('#footer-text-input').val();
        document.getElementById('invoice-footer-display').textContent = footerVal && footerVal.trim() ? footerVal : 'Invoice received from (supplier name) by our system Bizmart';
        const companyNameVal = $('#company-name-input').val();
        document.getElementById('invoice-company-name').textContent = companyNameVal || t('Company name', "Nom de l'Entreprise");
        document.getElementById('invoice-company-address').innerHTML = ($('#address-input').val() || t('Address not provided', 'Adresse non renseignee')).replace(/\n/g, '<br>');
        
        const regNumsText = "RC: " + ($('#rc-input').val() || 'N/A') + " | NIF: " + ($('#nif-input').val() || 'N/A') + " | NIS: " + ($('#nis-input').val() || 'N/A');
        document.getElementById('invoice-reg-numbers').textContent = regNumsText;
        
        const logoUrl = $('#logo-url-input').val();
        if (logoUrl) {
            $('#invoice-logo-preview').attr('src', logoUrl).show();
        } else {
            $('#invoice-logo-preview').hide();
        }
        
        // Footer preview fallback uses supplier-centric wording
        modal.style.display = 'flex';
    }

    function closeModal() {
        modal.style.display = 'none';
    }

    $('[data-segment]').on('click', function(e) {
        e.preventDefault();
        addSegment($(this).data('segment'));
    });
    
    $('#preview-test-invoice').on('click', function(e) {
        e.preventDefault();
        showModal();
    });

    $('#close-modal-btn').on('click', closeModal);
    
    $(modal).on('click', function(e) {
        if (e.target === modal) {
            closeModal();
        }
    });

    renderSegments();
    
    // Subcategories Management
    function updateSubcategoriesData() {
        const data = {};
        $('#subcategories-manager > div').each(function() {
            const category = $(this).data('category');
            const subs = [];
            $(this).find('.subcat-item').each(function() {
                const $text = $(this).find('.subcat-text');
                const raw = ($text.data('subcatValue') || '').toString().trim();
                subs.push(raw ? raw : $text.text().trim());
            });
            data[category] = subs;
        });
        $('#biz-subcategories-data').val(JSON.stringify(data));
    }
    
    // Add subcategory
    $(document).on('click', '.add-subcat-btn', function() {
        const category = $(this).data('category');
        const subcatName = prompt(t('Subcategory name:', 'Nom de la sous-categorie:'));
        if (!subcatName || !subcatName.trim()) return;
        
        const $container = $(this).closest('.biz-card').find('.subcategories-list');
        const newItem = `
            <div class="subcat-item" style="display: flex; align-items: center; gap: 8px; background: var(--bg-body); border: 1px solid var(--border-color); border-radius: 20px; padding: 6px 14px;">
                <span class="subcat-text" style="font-size: 13px;" data-subcat-value="${subcatName.trim()}">${subcatName.trim()}</span>
                <button type="button" class="delete-subcat-btn" style="border: none; background: none; color: var(--danger); cursor: pointer; font-size: 18px; line-height: 1;" title="${t('Delete', 'Supprimer')}">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        $container.append(newItem);
        updateSubcategoriesData();
    });
    
    // Delete subcategory
    $(document).on('click', '.delete-subcat-btn', function() {
        if (confirm(t('Delete this subcategory?', 'Supprimer cette sous-categorie?'))) {
            $(this).closest('.subcat-item').remove();
            updateSubcategoriesData();
        }
    });
    
    // Initial data sync
    updateSubcategoriesData();

    // ----------------------------
    // Import panel actions
    // ----------------------------
    function showImportMessage(msg, type = 'info') {
        const $area = $('#biz-import-status');
        $area.text(msg);
        if (type === 'error') $area.css('color', 'red'); else $area.css('color', '');
    }

    function extractAjaxError(xhr) {
        try {
            const json = xhr && xhr.responseJSON ? xhr.responseJSON : null;
            if (json && json.data) {
                if (typeof json.data === 'string') return json.data;
                if (json.data.message) return json.data.message;
                return JSON.stringify(json.data);
            }
        } catch (e) {}
        const status = xhr && xhr.status ? xhr.status : 0;
        const text = xhr && xhr.responseText ? xhr.responseText : '';
        const snippet = text && text.length > 300 ? (text.substring(0, 300) + '...') : text;
        return (status ? ('HTTP ' + status + ': ') : '') + (snippet || 'Request failed');
    }

    function normalizeErrorData(data) {
        if (!data) return 'Error';
        if (typeof data === 'string') return data;
        if (data.message) return data.message;
        try { return JSON.stringify(data); } catch (e) { return 'Error'; }
    }

    function renderConflicts(job) {
        const conflicts = job && Array.isArray(job.conflicts) ? job.conflicts : [];
        const $area = $('#biz-import-conflicts-area');
        const $tbody = $('#biz-import-conflicts-body');
        $tbody.empty();

        if (!conflicts.length) {
            $area.hide();
            return;
        }

        conflicts.forEach(function(c) {
            const pid = c && c.id ? parseInt(c.id, 10) : 0;
            if (!pid) return;

            const title = (c.title || '').toString();
            const sku = (c.sku || '').toString();
            const type = (c.type || '').toString();
            const existing = (c.existing !== undefined && c.existing !== null) ? String(c.existing) : '';
            const source = (c.source !== undefined && c.source !== null) ? String(c.source) : ((c.source_normalized !== undefined && c.source_normalized !== null) ? String(c.source_normalized) : '');
            const selling = (c.selling !== undefined && c.selling !== null) ? String(c.selling) : '';
            const editUrl = (c.edit_url || (editPostUrlBase + '?post=' + pid + '&action=edit'));

            const label = title ? (title + ' (#' + pid + ')') : ('#' + pid);
            const row = `
                <tr class="border-b border-gray-100 text-sm">
                    <td class="py-2 px-3">
                        <div style="font-weight:600;"><a href="${editUrl}" target="_blank" rel="noopener">${$('<div/>').text(label).html()}</a></div>
                        <div style="font-size:12px;color:#6b7280;">${$('<div/>').text(type || '').html()}</div>
                    </td>
                    <td class="py-2 px-3">${$('<div/>').text(sku).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(existing).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(source).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(selling).html()}</td>
                    <td class="py-2 px-3">
                        <label style="margin-right:10px;"><input type="radio" name="decision_${pid}" value="replace"> Replace</label>
                        <label><input type="radio" name="decision_${pid}" value="skip"> Skip</label>
                    </td>
                </tr>
            `;
            $tbody.append(row);
        });

        $area.show();
    }

    function renderPreview(job) {
        const items = job && Array.isArray(job.processed_preview) ? job.processed_preview : [];
        const $area = $('#biz-import-preview-area');
        const $tbody = $('#biz-import-preview-body');
        $tbody.empty();

        if (!items.length) {
            $area.hide();
            return;
        }

        // Show last 50 entries (already capped server-side but keep UI light)
        const slice = items.slice(Math.max(0, items.length - 50));
        slice.forEach(function(it) {
            const pid = it && it.id ? parseInt(it.id, 10) : 0;
            if (!pid) return;
            const title = (it.title || '').toString();
            const sku = (it.sku || '').toString();
            const type = (it.type || '').toString();
            const purchase = (it.purchase !== undefined && it.purchase !== null) ? String(it.purchase) : '';
            const existing = (it.prev_purchase !== undefined && it.prev_purchase !== null) ? String(it.prev_purchase) : ((it.existing_purchase !== undefined && it.existing_purchase !== null) ? String(it.existing_purchase) : '');
            const selling = (it.selling !== undefined && it.selling !== null) ? String(it.selling) : '';
            const action = (it.action || '').toString();
            const editUrl = (it.edit_url || (editPostUrlBase + '?post=' + pid + '&action=edit'));

            const label = title ? (title + ' (#' + pid + ')') : ('#' + pid);
            const badge = action ? `<span style="display:inline-block;padding:2px 8px;border-radius:999px;background:#eef2ff;color:#3730a3;font-size:12px;font-weight:600;">${$('<div/>').text(action).html()}</span>` : '';
            const row = `
                <tr class="border-b border-gray-100 text-sm">
                    <td class="py-2 px-3"><a href="${editUrl}" target="_blank" rel="noopener">${$('<div/>').text(label).html()}</a></td>
                    <td class="py-2 px-3">${$('<div/>').text(sku).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(type).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(purchase).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(existing).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(selling).html()}</td>
                    <td class="py-2 px-3">${badge}</td>
                </tr>
            `;
            $tbody.append(row);
        });

        $area.show();
    }

    function biz_get_count() {
        const meta = $('#import-meta-key').val() || '';
        if (!meta) { showImportMessage('Please enter a meta key', 'error'); return; }
        showImportMessage('Checking...');
        $.post(ajaxUrl, { action: 'bizmart_final_import_get_count', meta_key: meta, nonce: ajaxNonce }, function(r){
            if (!r || !r.success) { showImportMessage(normalizeErrorData(r && r.data), 'error'); return; }
            const d = r.data;
            const usedKey = d && d.used_meta_key ? (' Used key: ' + d.used_meta_key + '.') : '';
            showImportMessage('Found: ' + d.total_non_empty + ' (products: ' + d.products + ', variations: ' + d.variations + '). Total exists: ' + d.total_exists + '.' + usedKey);

            // Also fetch a sample list of matching products to make this useful.
            biz_preview_items(meta);
        }).fail(function(xhr){ showImportMessage(extractAjaxError(xhr), 'error'); });
    }

    function renderAvailableItems(payload) {
        const $area = $('#biz-import-available-area');
        const $tbody = $('#biz-import-available-body');
        const $note = $('#biz-import-available-note');
        $tbody.empty();

        const usedKey = payload && payload.used_meta_key ? payload.used_meta_key : '';
        const items = payload && Array.isArray(payload.items) ? payload.items : [];
        const limit = payload && payload.limit ? payload.limit : items.length;

        if (!items.length) {
            $area.hide();
            return;
        }

        $note.text('Showing ' + items.length + ' sample items (limit ' + limit + '). ' + (usedKey ? ('Used key: ' + usedKey) : ''));

        items.forEach(function(it){
            const pid = it && it.id ? parseInt(it.id, 10) : 0;
            if (!pid) return;
            const title = (it.title || '').toString();
            const sku = (it.sku || '').toString();
            const type = (it.type || '').toString();
            const raw = (it.source_raw !== undefined && it.source_raw !== null) ? String(it.source_raw) : '';
            const norm = (it.source_normalized !== undefined && it.source_normalized !== null) ? String(it.source_normalized) : '';
            const existing = (it.existing_purchase !== undefined && it.existing_purchase !== null) ? String(it.existing_purchase) : '';
            const selling = (it.selling !== undefined && it.selling !== null) ? String(it.selling) : '';
            const editUrl = (it.edit_url || (editPostUrlBase + '?post=' + pid + '&action=edit'));
            const label = title ? (title + ' (#' + pid + ')') : ('#' + pid);

            const row = `
                <tr class="border-b border-gray-100 text-sm">
                    <td class="py-2 px-3"><a href="${editUrl}" target="_blank" rel="noopener">${$('<div/>').text(label).html()}</a></td>
                    <td class="py-2 px-3">${$('<div/>').text(sku).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(type).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(raw).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(norm).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(existing).html()}</td>
                    <td class="py-2 px-3">${$('<div/>').text(selling).html()}</td>
                </tr>
            `;
            $tbody.append(row);
        });

        $area.show();
        applyAvailableFilter();
    }

    function applyAvailableFilter() {
        const q = ($('#biz-import-available-filter').val() || '').toString().trim().toLowerCase();
        const $rows = $('#biz-import-available-body tr');
        if (!q) { $rows.show(); return; }
        $rows.each(function(){
            const text = $(this).text().toLowerCase();
            $(this).toggle(text.indexOf(q) !== -1);
        });
    }

    $('#biz-import-available-filter').on('input', function(){ applyAvailableFilter(); });

    function biz_preview_items(meta) {
        meta = (meta || $('#import-meta-key').val() || '').toString();
        if (!meta) return;
        $.post(ajaxUrl, { action: 'bizmart_final_import_preview_items', meta_key: meta, limit: 30, nonce: ajaxNonce }, function(r){
            if (!r || !r.success) {
                $('#biz-import-available-area').hide();
                return;
            }
            renderAvailableItems(r.data);
        }).fail(function(){ $('#biz-import-available-area').hide(); });
    }

    function renderImportSummary(job) {
        const $wrap = $('#biz-import-summary');
        const $txt = $('#biz-import-summary-text');
        const $bar = $('#biz-import-progress-bar');
        const $pct = $('#biz-import-progress-left');
        const $frac = $('#biz-import-progress-right');

        if (!job) { $wrap.hide(); return; }
        const total = parseInt(job.total || 0, 10) || 0;
        const offset = parseInt(job.offset || 0, 10) || 0;
        const processed = parseInt(job.processed || 0, 10) || 0;
        const replaced = parseInt(job.replaced || 0, 10) || 0;
        const skipped = parseInt(job.skipped || 0, 10) || 0;
        const status = (job.status || 'unknown').toString();
        const dry = job.dry_run ? ' (dry-run)' : '';
        const pct = total > 0 ? Math.min(100, Math.round((offset / total) * 100)) : 0;

        $txt.text('Status: ' + status + dry + ' - processed: ' + processed + ', replaced: ' + replaced + ', skipped: ' + skipped);
        $bar.css('width', pct + '%');
        $pct.text(pct + '%');
        $frac.text(offset + ' / ' + total);
        $wrap.show();
    }

    function biz_start_import() {
        const meta = $('#import-meta-key').val() || '';
        const policy = $('#import-policy').val() || 'prompt';
        if (!meta) { showImportMessage('Please enter a meta key', 'error'); return; }
        showImportMessage('Starting import...');
        $.post(ajaxUrl, { action: 'bizmart_final_start_import_job', meta_key: meta, policy: policy, nonce: ajaxNonce }, function(r){
            if (!r || !r.success) { showImportMessage(normalizeErrorData(r && r.data), 'error'); return; }
            showImportMessage('Import started - it will continue in the background even if you leave this page.');
            renderImportSummary(r.data);
            biz_preview_items(meta);
            pollImportStatus();
        }).fail(function(xhr){ showImportMessage(extractAjaxError(xhr), 'error'); });
    }

    function pollImportStatus(interval = 1500) {
        clearTimeout(window._biz_import_poll);
        $.post(ajaxUrl, { action: 'bizmart_final_get_import_job_status', nonce: ajaxNonce }, function(r){
            if (!r || !r.success) {
                showImportMessage('No job');
                renderImportSummary(null);
                biz_preview_items($('#import-meta-key').val() || '');
                return;
            }
            const job = r.data;
            if (!job) {
                showImportMessage('No active job');
                renderImportSummary(null);
                biz_preview_items($('#import-meta-key').val() || '');
                $('#biz-import-json').val('');
                renderPreview(null);
                renderConflicts(null);
                return;
            }
            $('#biz-import-json').val(JSON.stringify(job, null, 2));
            showImportMessage('Status: ' + (job.status || 'unknown') + ' - processed: ' + (job.processed||0) + ' / ' + (job.total||0));

            renderImportSummary(job);
            // Keep the available list visible while running (helps verify mapping)
            biz_preview_items($('#import-meta-key').val() || (job.meta_key || ''));

            renderPreview(job);
            renderConflicts(job);

            // if paused_for_confirmation, show resume UI
            if (job.status === 'paused_for_confirmation') {
                $('#biz-import-resume-area').show();
            } else {
                $('#biz-import-resume-area').hide();
            }

            if (job.status && ['completed','cancelled'].indexOf(job.status) === -1) {
                window._biz_import_poll = setTimeout(function(){ pollImportStatus(interval); }, interval);
            }
        }).fail(function(xhr){ showImportMessage(extractAjaxError(xhr), 'error'); });
    }

    function biz_resume_import(action_choice, decisions) {
        const data = { action: 'bizmart_final_resume_import_job', action_choice: action_choice, nonce: ajaxNonce };
        if (decisions) data.decisions = JSON.stringify(decisions);
        $.post(ajaxUrl, data, function(r){
            if (!r || !r.success) { showImportMessage(normalizeErrorData(r && r.data), 'error'); return; }
            showImportMessage('Resumed. Polling...');
            pollImportStatus();
        }).fail(function(xhr){ showImportMessage(extractAjaxError(xhr), 'error'); });
    }

    // Cancel import UI removed; cancellation is not exposed to admins via the settings UI.

    /* Cleanup (delete source meta) client helpers */
    function renderCleanupSummary(job) {
        const $wrap = $('#biz-cleanup-summary');
        if (!job) { $wrap.hide(); return; }
        const total = parseInt(job.total || 0, 10) || 0;
        const offset = parseInt(job.offset || 0, 10) || 0;
        const processed = parseInt(job.processed || 0, 10) || 0;
        const status = (job.status || 'unknown').toString();
        const pct = total > 0 ? Math.min(100, Math.round((offset / total) * 100)) : 0;
        $('#biz-cleanup-status').text('Status: ' + status + ' - processed: ' + processed);
        $('#biz-cleanup-progress-bar').css('width', pct + '%');
        $('#biz-cleanup-pct').text(pct + '%');
        $('#biz-cleanup-fraction').text(offset + ' / ' + total);
        $wrap.show();
    }

    function biz_start_cleanup() {
        const meta = $('#import-meta-key').val() || '';
        if (!meta) { alert('Please enter a meta key to delete.'); return; }
        if (!confirm('Are you sure? This will permanently delete the chosen meta values for all products.')) return;
        $.post(ajaxUrl, { action: 'bizmart_final_start_cleanup_job', meta_key: meta, nonce: ajaxNonce }, function(r){
            if (!r || !r.success) { alert('Failed to start cleanup: ' + JSON.stringify(r && r.data)); return; }
            renderCleanupSummary(r.data);
            pollCleanupStatus();
        }).fail(function(xhr){ alert('Cleanup start failed: ' + extractAjaxError(xhr)); });
    }

    function downloadCsv(filename, text) {
        const blob = new Blob([text], { type: 'text/csv;charset=utf-8;' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        setTimeout(function(){ document.body.removeChild(a); URL.revokeObjectURL(url); }, 500);
    }

    function exportImportAudit() {
        $.post(ajaxUrl, { action: 'bizmart_final_export_import_audit', nonce: ajaxNonce }, function(r){
            if (!r || !r.success) { alert('No import audit available'); return; }
            const csv = r.data && r.data.csv ? r.data.csv : '';
            if (!csv) { alert('No audit data'); return; }
            downloadCsv('bizmart-import-audit.csv', csv);
        }).fail(function(xhr){ alert('Export failed: ' + extractAjaxError(xhr)); });
    }

    function exportCleanupAudit() {
        $.post(ajaxUrl, { action: 'bizmart_final_export_cleanup_audit', nonce: ajaxNonce }, function(r){
            if (!r || !r.success) { alert('No cleanup audit available'); return; }
            const csv = r.data && r.data.csv ? r.data.csv : '';
            if (!csv) { alert('No audit data'); return; }
            downloadCsv('bizmart-cleanup-audit.csv', csv);
        }).fail(function(xhr){ alert('Export failed: ' + extractAjaxError(xhr)); });
    }

    function pollCleanupStatus(interval = 2000) {
        clearTimeout(window._biz_cleanup_poll);
        $.post(ajaxUrl, { action: 'bizmart_final_get_cleanup_job_status', nonce: ajaxNonce }, function(r){
            if (!r || !r.success) { renderCleanupSummary(null); return; }
            const job = r.data;
            if (!job) { renderCleanupSummary(null); return; }
            renderCleanupSummary(job);
            if (job.status && ['completed','cancelled'].indexOf(job.status) === -1) {
                window._biz_cleanup_poll = setTimeout(function(){ pollCleanupStatus(interval); }, interval);
            }
        }).fail(function(){ renderCleanupSummary(null); });
    }

    // Cancel cleanup UI removed; cancellation not exposed via settings UI.

    // Bind UI
    $('#biz-get-import-count').on('click', function(e){ e.preventDefault(); biz_get_count(); });
    $('#biz-start-import').on('click', function(e){ e.preventDefault(); biz_start_import(); });
    // 'Cancel import' control removed from UI.
    $('#biz-resume-replace-all').on('click', function(e){ e.preventDefault(); biz_resume_import('replace_all'); });
    $('#biz-resume-skip-all').on('click', function(e){ e.preventDefault(); biz_resume_import('skip_all'); });
    $('#biz-start-cleanup').on('click', function(e){ e.preventDefault(); biz_start_cleanup(); });
    // 'Cancel cleanup' control removed from UI.
    $('#biz-export-import-audit').on('click', function(e){ e.preventDefault(); exportImportAudit(); });
    $('#biz-export-cleanup-audit').on('click', function(e){ e.preventDefault(); exportCleanupAudit(); });

    // Auto-resume: if a job is running in the background, show progress immediately
    pollImportStatus(1500);

    // Preset dropdown -> populate meta key
    $('#import-meta-preset').on('change', function() {
        const val = ($(this).val() || '').toString().trim();
        if (!val) return;
        if (val === '_custom') {
            $('#import-meta-key').val('');
            $('#import-meta-key').focus();
            updateDetectedIndicator('');
            return;
        }
        $('#import-meta-key').val(val);
        // Immediately show counts to confirm selection
        updateDetectedIndicator(val);
        biz_get_count();
    });

    // Update detected indicator when typing or selecting
    function updateDetectedIndicator(val) {
        val = (val || '').toString().trim();
        if (!val) { $('#biz-preset-detected-indicator').hide(); return; }
        if (Array.isArray(bizDetectedMetaKeys) && bizDetectedMetaKeys.indexOf(val) !== -1) {
            $('#biz-preset-detected-indicator').show();
        } else {
            $('#biz-preset-detected-indicator').hide();
        }
    }

    $('#import-meta-key').on('input', function() { updateDetectedIndicator($(this).val()); });

    $('#biz-submit-decisions').on('click', function(e){
        e.preventDefault();
        const jobText = $('#biz-import-json').val();
        let job = null;
        try { job = jobText ? JSON.parse(jobText) : null; } catch (err) { job = null; }
        const conflicts = job && Array.isArray(job.conflicts) ? job.conflicts : [];
        if (!conflicts.length) {
            showImportMessage('No conflicts to decide', 'error');
            return;
        }

        const decisions = {};
        conflicts.forEach(function(c) {
            const pid = c && c.id ? parseInt(c.id, 10) : 0;
            if (!pid) return;
            const val = $('input[name="decision_' + pid + '"]:checked').val();
            if (val === 'replace' || val === 'skip') decisions[String(pid)] = val;
        });

        if (Object.keys(decisions).length === 0) {
            showImportMessage('Select Replace/Skip for at least one item', 'error');
            return;
        }
        biz_resume_import('decide', decisions);
    });

    // Modern tab navigation
    jQuery(document).ready(function($) {
        // Tab switching
        $('.biz-nav-item').on('click', function() {
            const section = $(this).data('section');
            
            // Update nav items
            $('.biz-nav-item').removeClass('active');
            $(this).addClass('active');
            
            // Update sections
            $('.biz-section').removeClass('active');
            $('.biz-section[data-section="' + section + '"]').addClass('active');
            
            // Save to localStorage
            localStorage.setItem('bizmart_active_tab', section);
        });
        
        // Restore last tab
        const lastTab = localStorage.getItem('bizmart_active_tab');
        if (lastTab) {
            $('.biz-nav-item[data-section="' + lastTab + '"]').trigger('click');
        }
        
        // Toggle switches: render compact label + accessible toggle
        $('input[type="checkbox"]').each(function() {
            const checkbox = $(this);
            const wrapper = $('<div class="biz-toggle-group"></div>');

            // Ensure the checkbox has an id so the label 'for' attribute can target it
            let cbId = checkbox.attr('id');
            if (!cbId) {
                cbId = 'biz-toggle-' + Math.random().toString(36).slice(2,9);
                checkbox.attr('id', cbId);
            }

            // Find the original label and extract its inner content (icon + text)
            const $originalLabel = checkbox.closest('.biz-field-group, div').find('.biz-field-label, label').first();
            const labelContent = $originalLabel.length ? $originalLabel.html() : checkbox.attr('name') || '';
            $originalLabel.hide(); // hide original to avoid duplicate text

            // Localized short note using admin language
            const noteText = '<?php echo esc_js( function_exists("bizmart_text") ? bizmart_text("Show / Hide on invoice", "Afficher / Masquer sur la facture") : "Show / Hide on invoice" ); ?>';

            // Create clickable label element tied to the checkbox (main + note)
            const labelEl = $('<label class="biz-toggle-label" for="' + cbId + '"></label>');
            labelEl.html('<div class="biz-toggle-main">' + labelContent + '</div><div class="biz-toggle-note">' + noteText + '</div>');

            // Build a simple toggle with a knob for animation
            const toggle = $('<div class="biz-toggle" role="switch" aria-checked="' + (checkbox.is(':checked') ? 'true' : 'false') + '"></div>');
            const knob = $('<div class="biz-toggle-knob"></div>');
            toggle.append(knob);
            if (checkbox.is(':checked')) toggle.addClass('on');

            // Interaction: clicking label or toggle toggles the underlying checkbox
            const toggleAction = function() {
                checkbox.prop('checked', !checkbox.is(':checked')).trigger('change');
            };
            toggle.on('click', toggleAction);
            labelEl.on('click', function(e) {
                // allow native label behavior but ensure change event fires
                e.preventDefault();
                toggleAction();
            });

            // Keep toggle state synced when the checkbox value changes programmatically
            checkbox.on('change', function() {
                const on = checkbox.is(':checked');
                toggle.toggleClass('on', on);
                toggle.attr('aria-checked', on ? 'true' : 'false');
            });

            // Render: label then toggle for compact inline look
            wrapper.append(labelEl).append(toggle);
            checkbox.hide().before(wrapper);
        });
    });
});

// v3.0: Valuation radio card styling
(function(){
    function syncValCards() {
        document.querySelectorAll('.biz-val-radio').forEach(function(r) {
            var card = r.closest('label').querySelector('.biz-val-card');
            if (r.checked) {
                card.style.borderColor = 'var(--primary)';
                card.style.background = 'linear-gradient(135deg, #eff6ff, #dbeafe)';
                card.style.boxShadow = '0 0 0 3px rgba(79,70,229,0.15)';
            } else {
                card.style.borderColor = 'var(--gray-200)';
                card.style.background = 'white';
                card.style.boxShadow = 'none';
            }
        });
    }
    document.querySelectorAll('.biz-val-radio').forEach(function(r) {
        r.addEventListener('change', syncValCards);
    });
    syncValCards();
})();
</script>

        <!-- Floating Save Button -->
        <div class="biz-save-float">
            <button type="button" onclick="(function(){var f=document.getElementById('biz-modern-form'); if(!f) return; var h=document.getElementById('biz_save_settings_input'); if(!h){ h=document.createElement('input'); h.type='hidden'; h.name='biz_save_settings'; h.id='biz_save_settings_input'; h.value='1'; f.appendChild(h);} else { h.value='1'; } f.submit(); })();" class="biz-btn-save" aria-label="Save changes">
                <i class="fas fa-save" aria-hidden="true"></i>
                <?php echo esc_html(function_exists('bizmart_text') ? bizmart_text('Save Changes', 'Enregistrer', 'حفظ التغييرات') : 'Save Changes'); ?>
            </button>
        </div>

    </form>
    </div>








